from datetime import datetime
from core.db.mongo import users_collection, login_attempts_collection
from bson import ObjectId

def create_user(user_data: dict):
    user_data.update({
        "is_active": True,
        "is_verified": False,
        "dob": None,
        "gender" : None,
        "primary_email_verified" : False,
        "dob_verified": False,
        "recovery_phone": None,
        "recovery_phone_verified" : False,
        "recovery_email": None,
        "recovery_email_verified" : False,
        "address": None,
        "created_at": datetime.utcnow(),
        "updated_at": datetime.utcnow(),
        "connected_apps": [],
        "is_admin":False,        
    })
    result = users_collection.insert_one(user_data)
    return str(result.inserted_id)

def find_user_by_email(email: str):
    return users_collection.find_one({"primary_email": email})

def find_user_by_phone(phone: str):
    return users_collection.find_one({"phone_number": phone})

def find_user_by_identifier(identifier: str):
    return users_collection.find_one({
        "$or": [
            {"primary_email": identifier.lower()},
            {"phone_number": identifier},
            # {"recovery_email" : identifier.lower()},
            # {"recovery_phone", identifier}
        ]
    })
    
    
def find_user_by_id(user_id: str):
    return users_collection.find_one(
        {"_id": ObjectId(user_id)}
    )
    
    
def mark_user_verified(user_id: str):
    users_collection.update_one(
        {"_id": ObjectId(user_id)},
        {"$set": {"is_verified": True}}
    )
    
    



ALLOWED_FIELDS = {
    "first_name",
    "last_name",
    "phone_number",
    "primary_email",
    "recovery_email",
    "recovery_phone",
    "address",
    "dob",
    "gender"
}


def update_user(user_id: str, update_data: dict):
    print(update_data)
    update_fields = {
        key: value
        for key, value in update_data.items()
        if key in ALLOWED_FIELDS
        # if key in ALLOWED_FIELDS and value is not None
        
    }
    print(update_fields)
    if not update_fields:
        return False

    update_fields["updated_at"] = datetime.utcnow()

    result = users_collection.update_one(
        {"_id": ObjectId(user_id)},
        {"$set": update_fields}
    )

    return result.modified_count > 0




from core.db.mongo import users_collection


def find_user_by_identifier_and_dob(identifier: str, dob: str):
    """
    identifier: email or phone
    dob: YYYY-MM-DD (string)
    """
    return users_collection.find_one({
        "$and": [
            {
                "$or": [
                    {"primary_email": identifier.lower()},
                    {"phone_number": identifier}
                ]
            },
            {"dob": dob}
        ]
    })



def find_user_by_any_identifier(identifier: str):
    """
    Finds user by:
    - primary email
    - primary phone
    - recovery email
    - recovery phone
    """

    query = {
        "$or": [
            {"primary_email": identifier.lower()},
            {"phone_number": identifier},
            {"recovery_email": identifier.lower()},
            {"recovery_phone": identifier},
        ]
    }

    return users_collection.find_one(query)




MAX_LOGIN_ATTEMPTS = 5
LOGIN_LOCK_HOURS = 1
LOGIN_LOCK_MINUTES = 1
from datetime import datetime, timedelta, timezone

# def can_attempt_login(user_id, ip):
#     print("Can attempt otp")
#     now = datetime.now(timezone.utc)

#     record = login_attempts_collection.find_one({
#         "user_id": ObjectId(user_id),
#         "ip_address": ip
#     })

#     if not record:
#         return True, None

#     locked_until = record.get("locked_until")
#     if locked_until:
#         if locked_until > now:
#             return False, f"Account locked due to multiple failed attempts. Try again after {LOGIN_LOCK_HOURS} hours."
#         else:
#             # 🔥 Lock expired → reset
#             login_attempts_collection.delete_one({"_id": record["_id"]})
#             return True, None

#     return True, None


from datetime import datetime, timezone

def can_attempt_login(user_id, ip):
    now = datetime.now(timezone.utc)

    record = login_attempts_collection.find_one({
        "user_id": ObjectId(user_id),
        "ip_address": ip
    })

    if not record:
        return True, None

    locked_until = record.get("locked_until")

    # 🔧 Normalize locked_until
    if locked_until:
        if locked_until.tzinfo is None:
            locked_until = locked_until.replace(tzinfo=timezone.utc)

        # 🔒 Still locked
        if locked_until > now:
            return False, "Too many failed attempts. Try again later."

        # 🔓 Lock expired → cleanup
        login_attempts_collection.delete_one({"_id": record["_id"]})
        return True, None

    return True, None


# def register_failed_login(user_id, ip):
#     now = datetime.now(timezone.utc)

#     record = login_attempts_collection.find_one({
#         "user_id": ObjectId(user_id),
#         "ip_address": ip
#     })

#     if not record:
#         login_attempts_collection.insert_one({
#             "user_id": ObjectId(user_id),
#             "ip_address": ip,
#             "attempts": 1,
#             "first_attempt_at": now,
#             "last_attempt_at": now,
#             "locked_until": None
#         })
#         return

#     attempts = record["attempts"] + 1

#     if attempts >= MAX_LOGIN_ATTEMPTS:
#         login_attempts_collection.update_one(
#             {"_id": record["_id"]},
#             {"$set": {
#                 "attempts": attempts,
#                 "locked_until": now + timedelta(hours=LOGIN_LOCK_HOURS),
#                 "last_attempt_at": now
#             }}
#         )
#     else:
#         login_attempts_collection.update_one(
#             {"_id": record["_id"]},
#             {"$inc": {"attempts": 1},
#              "$set": {"last_attempt_at": now}}
#         )


def register_failed_login(user_id, ip):
    now = datetime.now(timezone.utc)

    record = login_attempts_collection.find_one({
        "user_id": ObjectId(user_id),
        "ip_address": ip
    })

    if not record:
        login_attempts_collection.insert_one({
            "user_id": ObjectId(user_id),
            "ip_address": ip,
            "attempts": 1,
            "first_attempt_at": now,
            "last_attempt_at": now,
            "locked_until": None
        })
        return

    attempts = record["attempts"] + 1

    update = {
        "attempts": attempts,
        "last_attempt_at": now
    }

    if attempts >= MAX_LOGIN_ATTEMPTS:
        update["locked_until"] = now + timedelta(minutes=LOGIN_LOCK_MINUTES)

    login_attempts_collection.update_one(
        {"_id": record["_id"]},
        {"$set": update}
    )


def clear_login_attempts(user_id, ip):
    login_attempts_collection.delete_one({
        "user_id": ObjectId(user_id),
        "ip_address": ip
    })
    
    
    


def add_connected_app(user_id, app_id):
    users_collection.update_one(
        {"_id": ObjectId(user_id)},
        {"$addToSet": {"connected_apps": app_id}}
    )
    

def remove_connected_app(user_id, app_id):
    users_collection.update_one(
        {"_id": ObjectId(user_id)},
        {"$pull": {"connected_apps": app_id}}
    )