from datetime import datetime, timezone
from core.db.typesense import get_indzs_typesense_client
import typesense

def utc_now():
    return datetime.now(timezone.utc).strftime("%Y-%m-%dT%H:%M:%SZ")

class IndzsAdapter:

    def __init__(self):
        self.client = get_indzs_typesense_client()
        self.collection = "users"

    def provision_user(self, user):
        """
        If user exists → update only isActive & emailVerified
        If not → create full document
        """
        doc_id = user["primary_email"]  # using email as document ID

        try:
            # 🔍 Check if document exists
            self.client.collections[self.collection].documents[doc_id].retrieve()

            # ✅ EXISTS → partial update only
            self.client.collections[self.collection].documents[doc_id].update({
                "isActive": True,
                "emailVerified": True,
                "updatedAt": utc_now(),
            })

        except typesense.exceptions.ObjectNotFound:
            # ❌ DOES NOT EXIST → create full document
            document = {
                "id": doc_id,
                "username" : user["primary_email"],
                "email": user["primary_email"],
                "password": user["chitraplay_password"],  # MUST be hashed
                "walletBalance": 0.0,
                "createdAt": utc_now(),
                "isActive": True,
                "emailVerified": True,
            }

            self.client.collections[self.collection].documents.upsert(document)

    def deactivate_user(self, user):
        doc_id = user["primary_email"]
        try:
            self.client.collections[self.collection].documents[doc_id].update({
                "isActive": False,
                "updatedAt": utc_now(),
            })
        except typesense.exceptions.ObjectNotFound:
            pass

    def update_password(self, user, app_password):
        # ❌ Not supported / not recommended for Typesense
        # raise Exception("Password update not supported for Typesense")
        pass

