# from django.shortcuts import render
import re

from django.core import exceptions
from apps.accounts.services.password_service import hash_password, bcrypt_password_hash, custom_bcrypt_password_hash
# Create your views here.
from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status

from apps.accounts.models.user import find_user_by_identifier, find_user_by_id, mark_user_verified, find_user_by_any_identifier, can_attempt_login, clear_login_attempts, register_failed_login
from apps.accounts.services.password_service import verify_password
from apps.auth.utils.jwt_service import generate_access_token, generate_refresh_token, verify_access_token, verify_refresh_token
from .serializer import serialize_user
from core.utils.validation import validate_name,validate_phone, validate_email_field, validate_identifier, validate_dob, validate_gender, validate_otp, validate_password, validate_country_code
from django.core.exceptions import ValidationError

from core.db.mongo import users_collection, otp_collection, user_sessions_collection
from .utils.validators import is_valid_email, is_valid_phone 
from bson import ObjectId
from .utils.device import parse_device
from datetime import datetime, timezone
from hashlib import sha256
from core.utils.audit import create_audit_log, AuditAction
from .decorators import jwt_required
from .utils.session import deactivate_session, hash_refresh_token, deactivate_all_sessions, deactivate_specific_device
from core.db.mongo import user_sessions_collection
from uuid import uuid4
from hashlib import sha256
from rest_framework.decorators import api_view, parser_classes
from rest_framework.parsers import MultiPartParser, FormParser
from rest_framework.response import Response
from django.core.files.storage import default_storage
from .utils.process_profile_image import process_profile_image
import os
from core.utils.geo import geo_lookup, get_geo_context
from core.utils.helpers import get_client_ip

from .models.otp import can_send_otp, save_otp, is_ip_blocked
from .utils.otp import generate_otp, hash_otp
from .models.otp import (
    get_valid_otp,
    increment_verify_attempt,
    mark_otp_verified,
    delete_all_user_otps,
    clear_ip_block,
    MAX_VERIFY_ATTEMPTS
)

from .utils.otp import verify_otp_hash
from .utils.email_service import send_otp_email, send_new_device_login_email
from .utils.sms_service import send_otp_sms
from config.settings.base import DEFAULT_FROM_EMAIL
from core.utils.helpers import env_bool



@api_view(["GET"])
def test_api(request):
    return Response(
        {"status": "Auth API working ✅"},
        status=status.HTTP_200_OK
    )




from apps.accounts.models.user import (
    create_user,
    find_user_by_email,
    find_user_by_phone
)

EMAIL_REGEX = r"^[\w\.-]+@[\w\.-]+\.\w+$"
PHONE_REGEX = r"^[6-9]\d{9}$"  # India-friendly


def safe_audit(**kwargs):
    try:
        create_audit_log(**kwargs)
    except Exception:
        pass
    
@api_view(["POST"])
def register_user(request):
    data = request.data


    # Mandatory fields
    required_fields = ["first_name", "phone_number", "primary_email", "password", "country_code"]
    for field in required_fields:
        if not data.get(field):
            return Response(
                {"error": f"{field} is required"},
                status=status.HTTP_400_BAD_REQUEST
            )

    try:
        validate_name(data["first_name"])
        validate_email_field(data["primary_email"])
        validate_phone(data["phone_number"])
        validate_password(data["password"])
        validate_country_code(data["country_code"])
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )

    if data.get("last_name"):
        try:
            validate_name(data["last_name"])
        except ValidationError:
            return Response(
                {"error": "Invalid Last Name"},
                status=status.HTTP_400_BAD_REQUEST
            )

    # Duplicate checks
    if find_user_by_email(data["primary_email"]):
        return Response(
            {"error": "Email already registered"},
            status=status.HTTP_409_CONFLICT
        )

    if find_user_by_phone(data["phone_number"]):
        return Response(
            {"error": "Phone number already registered"},
            status=status.HTTP_409_CONFLICT
        )

    hashed_password = hash_password(data["password"])
    hashed_chitraplay_password = bcrypt_password_hash(data["password"])
    hashed_kujanam_password = custom_bcrypt_password_hash(data["password"], 12)

    user_payload = {
        "first_name": data["first_name"].strip(),
        "last_name": data.get("last_name", "").strip(),
        "phone_number": f"""{data["phone_number"]}""",
        "primary_email": data["primary_email"].lower(),
        "recovery_phone": data.get("recovery_phone"),
        "recovery_email": data.get("recovery_email"),
        "password": hashed_password,
        "chitraplay_password": hashed_chitraplay_password,
        "bhulok_password": hashed_chitraplay_password,
        "kujanam_password": hashed_kujanam_password,
        "address": data.get("address"),
        "country_code":data.get("country_code")
    }

    user_id = create_user(user_payload)

    # ✅ AUDIT LOG (after successful creation)
    create_audit_log(
        action=AuditAction.USER_REGISTERED,
        actor_id=None,
        target_user_id=str(user_id),
        metadata={
            "registration_method": "password",
            "primary_email": data["primary_email"].lower()
        },
        request=request
    )

    return Response(
        {
            "message": "User registered successfully",
            "user_id": user_id
        },
        status=status.HTTP_201_CREATED
    )

@api_view(["POST"])
def login_user(request):
    data = request.data

    identifier = data.get("identifier")  # email or phone
    password = data.get("password")
    device_id = request.headers.get("X-Device-Id")
    user_agent = request.META.get("HTTP_USER_AGENT", "")
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)

    

    # 🔧 ENV FLAGS
    REQUIRE_DEVICE_ID = env_bool("REQUIRE_DEVICE_ID", True)
    REQUIRE_EMAIL_VERIFICATION = env_bool("REQUIRE_EMAIL_VERIFICATION", True)

    # ======================
    # Basic validations
    # ======================
    if REQUIRE_DEVICE_ID and not device_id:
        return Response(
            {"error": "Device ID missing"},
            status=status.HTTP_400_BAD_REQUEST
        )

    if not identifier or not password:
        return Response(
            {"error": "Identifier and password required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        validate_identifier(identifier)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )

    # ======================
    # Fetch user
    # ======================
    user = find_user_by_identifier(identifier)

    if not user:
        create_audit_log(
            action=AuditAction.LOGIN_FAILED,
            actor_id=None,
            metadata={
                "identifier": identifier,
                "device_id": device_id,
                "reason": "User not found"
            },
            request=request
        )
        return Response(
            {"error": "Invalid credentials"},
            status=status.HTTP_401_UNAUTHORIZED
        )

    user_id = str(user["_id"])

    # ======================
    # Brute-force protection
    # ======================
    allowed, error = can_attempt_login(user_id, ip)
    if not allowed:
        return Response({"error": error}, status=status.HTTP_403_FORBIDDEN)

    # ======================
    # Account state checks
    # ======================
    if not user.get("is_active", False):
        create_audit_log(
            action=AuditAction.LOGIN_FAILED,
            actor_id=user["_id"],
            metadata={
                "device_id": device_id,
                "reason": "Account suspended"
            },
            request=request
        )
        return Response(
            {"error": "Account has been suspended"},
            status=status.HTTP_403_FORBIDDEN
        )

    if REQUIRE_EMAIL_VERIFICATION and not user.get("is_verified", False):
        create_audit_log(
            action=AuditAction.LOGIN_FAILED,
            actor_id=user["_id"],
            metadata={
                "device_id": device_id,
                "reason": "Account not verified"
            },
            request=request
        )
        return Response(
            {"error": "Account not verified"},
            status=status.HTTP_403_FORBIDDEN
        )

    # ======================
    # Password verification
    # ======================
    if not verify_password(user["password"], password):
        register_failed_login(user_id, ip)

        create_audit_log(
            action=AuditAction.LOGIN_FAILED,
            actor_id=user["_id"],
            metadata={
                "device_id": device_id,
                "reason": "Invalid password"
            },
            request=request
        )

        return Response(
            {"error": "Invalid credentials"},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # ======================
    # Session & token logic
    # ======================
    session_id = str(uuid4())
    access_token = generate_access_token(user_id, session_id)
    refresh_token = generate_refresh_token(user_id, session_id)

    refresh_token_hash = sha256(refresh_token.encode()).hexdigest()
    device_info = parse_device(user_agent)

    now = datetime.now(timezone.utc)

    # 🔁 Deactivate previous sessions for same user + device
    user_sessions_collection.update_many(
        {
            "user_id": ObjectId(user_id),
            "device_id": device_id,
            "active": True,
            "is_active": True
        },
        {
            "$set": {
                "active": False,
                "last_active": now
            }
        }
    )

    # 🔥 Remove old records for same user + device
    user_sessions_collection.delete_many({
        "user_id": ObjectId(user_id),
        "device_id": device_id
    })

    # geo_data = (geo_lookup(ip)) or {}
    geo_data = get_geo_context(request)

    
    # ➕ Insert new session
    user_sessions_collection.insert_one({
        "user_id": ObjectId(user_id),
        "session_id": session_id,
        "device_id": device_id,
        "ip_address": ip,
        "browser": device_info.get("browser"),
        "os": device_info.get("os"),
        "user_agent": user_agent,
        "refresh_token_hash": refresh_token_hash,
        "created_at": now,
        "last_active": now,
        "is_active": True,
        "active": True,
        "identifier": identifier,
        "country" : geo_data.get("country",""),
        "country_name": geo_data.get("country_name", ""),
        "city": geo_data.get("city",""),
        "region" : geo_data.get("region",""),
        "lat" : geo_data.get("lat",""),
        "lng": geo_data.get("lng",""),
        "timezone": geo_data.get("timezone",""),
        "asn": geo_data.get("asn",""),
        "org":geo_data.get("org","")
        
    })
    
    clear_login_attempts(user_id, ip)

    location = f""" {geo_data.get("city","")} | {geo_data.get("region","")} | {geo_data.get("country_name", "")}"""

    # ======================
    # New Device Log in Alert
    # =======================
    if env_bool("ENABLE_LOGIN_ALERTS", False):
        try:
            send_new_device_login_email(
                to_email=user.get("primary_email"),
                browser= device_info.get("browser"),
                username=user.get("first_name"),
                device_name= f"""{device_info.get("os")} | {device_info.get("browser")} """ ,
                os_name=device_info.get("os"),
                user_agent=user_agent,
                location=location,
                ip_address=ip,
                sender_email=DEFAULT_FROM_EMAIL,
                sender_name=os.getenv("SECURITY_SENDER_NAME", "INDZS Security"))
            
        except Exception as e:
            print(e)
    
    # ======================
    # Audit success login
    # ======================
    create_audit_log(
        action=AuditAction.LOGIN,
        actor_id=user["_id"],
        metadata={
            "device_id": device_id
        },
        request=request
    )

    # ======================
    # Response + cookies
    # ======================
    response = Response(
        {
            "message": "Login successful",
            "user": {
                "id": user_id,
                "first_name": user["first_name"],
                "primary_email": user["primary_email"],
                "phone_number": user["phone_number"]
            }
        },
        status=status.HTTP_200_OK
    )

    response.set_cookie(
        key="access_token",
        value=access_token,
        httponly=True,
        secure=env_bool("JWT_COOKIE_SECURE", True),
        samesite=os.getenv("JWT_COOKIE_SAMESITE", "None"),
        max_age=int(os.getenv("JWT_ACCESS_TOKEN_EXP", 900))
    )

    response.set_cookie(
        key="refresh_token",
        value=refresh_token,
        httponly=True,
        secure=env_bool("JWT_COOKIE_SECURE", True),
        samesite=os.getenv("JWT_COOKIE_SAMESITE", "None"),
        max_age=int(os.getenv("JWT_REFRESH_TOKEN_EXP", 604800))
    )

    return response


@api_view(["GET"])
def list_logged_in_accounts(request):
    device_id = request.headers.get("X-Device-Id")

    REQUIRE_DEVICE_ID = env_bool("REQUIRE_DEVICE_ID", True)
    ENABLE_ACCOUNT_SWITCHER = env_bool("ENABLE_ACCOUNT_SWITCHER", True)

    if not ENABLE_ACCOUNT_SWITCHER:
        return Response(
            {"error": "Account switching disabled"},
            status=status.HTTP_403_FORBIDDEN
        )

    if REQUIRE_DEVICE_ID and not device_id:
        return Response(
            {"error": "Device ID missing"},
            status=status.HTTP_400_BAD_REQUEST
        )

    user_id = str(request.user.id)

    sessions = user_sessions_collection.find(
        {
            "device_id": device_id,
            "is_active": True
        }
    ).sort("last_active", -1)

    response_data = []
    for s in sessions:
        response_data.append({
            "user_id": str(s["user_id"]),
            "identifier": s.get("identifier"),
            "active": s.get("active", False),
            "last_active": s.get("last_active")
        })

    # 🧾 Audit log
    create_audit_log(
        action=AuditAction.VIEW_LOGGED_IN_ACCOUNTS,
        actor_id=user_id,
        metadata={
            "device_id": device_id,
            "accounts_found": len(response_data)
        },
        request=request
    )

    return Response(response_data, status=status.HTTP_200_OK)


@api_view(["POST"])
def switch_account(request):
    target_user_id = request.data.get("user_id")
    device_id = request.headers.get("X-Device-Id")
    refresh_token = request.COOKIES.get("refresh_token")

    ENABLE_ACCOUNT_SWITCHER = env_bool("ENABLE_ACCOUNT_SWITCHER",False)
    REQUIRE_DEVICE_ID = env_bool("REQUIRE_DEVICE_ID", False) 

    if not ENABLE_ACCOUNT_SWITCHER:
        return Response({"error": "Account switching disabled"}, status=403)

    if REQUIRE_DEVICE_ID and not device_id:
        return Response({"error": "Device ID missing"}, status=401)

    if not target_user_id or not refresh_token:
        return Response({"error": "Unauthorized"}, status=401)

    refresh_hash = sha256(refresh_token.encode()).hexdigest()
    actor_user_id = str(request.user.id)

    # 🔒 Verify current active session
    current_session = user_sessions_collection.find_one({
        "device_id": device_id,
        "refresh_token_hash": refresh_hash,
        "active": True,
        "is_active": True
    })

    if not current_session:
        create_audit_log(
            action=AuditAction.SWITCH_ACCOUNT_FAILED,
            actor_id=actor_user_id,
            metadata={
                "device_id": device_id,
                "reason": "Invalid or expired session"
            },
            request=request
        )
        return Response({"error": "Invalid session"}, status=401)

    # 🔒 Verify target session exists on same device
    target_session = user_sessions_collection.find_one({
        "device_id": device_id,
        "user_id": ObjectId(target_user_id),
        "is_active": True
    })

    if not target_session:
        create_audit_log(
            action=AuditAction.SWITCH_ACCOUNT_FAILED,
            actor_id=actor_user_id,
            metadata={
                "device_id": device_id,
                "target_user_id": target_user_id,
                "reason": "Target account not logged in on this device"
            },
            request=request
        )
        return Response(
            {"error": "Account not logged in on this device"},
            status=403
        )

    # 🔁 Deactivate current session
    user_sessions_collection.update_one(
        {"_id": current_session["_id"]},
        {"$set": {"active": False}}
    )

    # 🔁 Activate target session
    new_session_id = target_session["session_id"]
    access_token = generate_access_token(target_user_id, new_session_id)
    refresh_token = generate_refresh_token(target_user_id, new_session_id)

    user_sessions_collection.update_one(
        {"_id": target_session["_id"]},
        {
            "$set": {
                "active": True,
                "refresh_token_hash": sha256(refresh_token.encode()).hexdigest(),
                "last_active": datetime.now(timezone.utc)
            }
        }
    )

    # 🧾 Audit success
    create_audit_log(
        action=AuditAction.SWITCH_ACCOUNT,
        actor_id=actor_user_id,
        target_user_id=target_user_id,
        metadata={
            "device_id": device_id
        },
        request=request
    )

    response = Response(
        {
            "success": True,
            "switched_to": target_user_id
        },
        status=200
    )

    response.set_cookie(
        key="access_token",
        value=access_token,
        httponly=True,
        secure=env_bool("JWT_COOKIE_SECURE", True),
        samesite=os.getenv("JWT_COOKIE_SAMESITE", "Lax"),
        max_age=int(os.getenv("JWT_ACCESS_TOKEN_EXP", 900))
    )

    response.set_cookie(
        key="refresh_token",
        value=refresh_token,
        httponly=True,
        secure=env_bool("JWT_COOKIE_SECURE", True),
        samesite=os.getenv("JWT_COOKIE_SAMESITE", "Lax"),
        max_age=int(os.getenv("JWT_REFRESH_TOKEN_EXP", 604800))
    )

    return response


@api_view(["POST"])
def delete_account_from_device(request):
    try:
        target_user_id = request.data.get("user_id")
        device_id = request.headers.get("X-Device-Id")
        refresh_token = request.COOKIES.get("refresh_token")

        ENABLE_ACCOUNT_SWITCHER = env_bool("ENABLE_ACCOUNT_SWITCHER", True) 
        REQUIRE_DEVICE_ID = env_bool("REQUIRE_DEVICE_ID", True)

        if not ENABLE_ACCOUNT_SWITCHER:
            return Response({"error": "Account management disabled"}, status=403)

        if REQUIRE_DEVICE_ID and not device_id:
            return Response({"error": "Device ID missing"}, status=401)

        if not target_user_id or not refresh_token:
            return Response({"error": "Unauthorized"}, status=401)

        refresh_hash = sha256(refresh_token.encode()).hexdigest()
        actor_user_id = str(request.user.id)

        # 🔒 Verify current active session belongs to actor
        current_session = user_sessions_collection.find_one({
            "device_id": device_id,
            "refresh_token_hash": refresh_hash,
            "active": True,
            "is_active": True,
            # "user_id": ObjectId(actor_user_id)
        })

        if not current_session:
            create_audit_log(
                action=AuditAction.DELETE_ACCOUNT_FROM_DEVICE_FAILED,
                actor_id=actor_user_id,
                metadata={
                    "device_id": device_id,
                    "reason": "Invalid session"
                },
                request=request
            )
            return Response({"error": "Invalid session"}, status=403)

        # 🔒 Check target session exists on same device
        target_session = user_sessions_collection.find_one({
            "device_id": device_id,
            "user_id": ObjectId(target_user_id),
            "is_active": True
        })

        if not target_session:
            create_audit_log(
                action=AuditAction.DELETE_ACCOUNT_FROM_DEVICE_FAILED,
                actor_id=actor_user_id,
                metadata={
                    "device_id": device_id,
                    "reason": "Target account not found on device"
                },
                request=request
            )
            return Response(
                {"error": "Account not found on this device"},
                status=404
            )

        was_active = target_session.get("active", False)

        # 🗑️ Delete the target account session
        user_sessions_collection.delete_one({"_id": target_session["_id"]})

        # 🧾 Audit success deletion
        create_audit_log(
            action=AuditAction.DELETE_ACCOUNT_FROM_DEVICE,
            actor_id=actor_user_id,
            target_user_id=target_user_id,
            metadata={
                "device_id": device_id,
                "was_active": was_active
            },
            request=request
        )

        # 🔁 If deleted account was active → switch to next
        if was_active:
            next_session = user_sessions_collection.find_one(
                {
                    "device_id": device_id,
                    "is_active": True
                },
                sort=[("last_active", -1)]
            )

            if next_session:
                new_access = generate_access_token(
                    str(next_session["user_id"]),
                    next_session["session_id"]
                )
                new_refresh = generate_refresh_token(
                    str(next_session["user_id"]),
                    next_session["session_id"]
                )

                user_sessions_collection.update_one(
                    {"_id": next_session["_id"]},
                    {
                        "$set": {
                            "active": True,
                            "refresh_token_hash": sha256(new_refresh.encode()).hexdigest(),
                            "last_active": datetime.now(timezone.utc)
                        }
                    }
                )

                response = Response(
                    {"success": True, "switched": True},
                    status=200
                )

                response.set_cookie(
                    "access_token",
                    new_access,
                    httponly=True,
                    secure=env_bool("JWT_COOKIE_SECURE", True),
                    samesite=os.getenv("JWT_COOKIE_SAMESITE", "Lax"),
                    max_age=int(os.getenv("JWT_ACCESS_TOKEN_EXP", 900))
                )

                response.set_cookie(
                    "refresh_token",
                    new_refresh,
                    httponly=True,
                    secure=env_bool("JWT_COOKIE_SECURE", True),
                    samesite=os.getenv("JWT_COOKIE_SAMESITE", "Lax"),
                    max_age=int(os.getenv("JWT_REFRESH_TOKEN_EXP", 604800))
                )

                return response

            # 🚪 No accounts left → logout device
            response = Response(
                {"success": True, "logged_out": True},
                status=200
            )
            response.delete_cookie("access_token")
            response.delete_cookie("refresh_token")
            return response

        # ✅ Deleted non-active account
        return Response({"success": True}, status=200)
    except Exception as e:
        print(e)
        return Response({"success": False}, status=500)



@api_view(["POST"])
@jwt_required
def logout_account(request):
    device_id = request.headers.get("X-Device-Id")
    access_token = request.COOKIES.get("access_token")
    refresh_token = request.COOKIES.get("refresh_token")

    ENABLE_ACCOUNT_SWITCHER = env_bool("ENABLE_ACCOUNT_SWITCHER", True) 
    REQUIRE_DEVICE_ID = env_bool("REQUIRE_DEVICE_ID", True)

    if REQUIRE_DEVICE_ID and not device_id:
        return Response({"error": "Device ID missing"}, status=401)

    if not access_token or not refresh_token:
        return Response({"error": "Unauthorized"}, status=401)

    payload = verify_access_token(access_token)
    if not payload:
        return Response({"error": "Invalid token"}, status=401)

    user_id = payload["sub"]
    session_id = payload["session_id"]
    now = datetime.now(timezone.utc)

    refresh_hash = sha256(refresh_token.encode()).hexdigest()

    # 🔒 Verify current session ownership
    current_session = user_sessions_collection.find_one({
        "user_id": ObjectId(user_id),
        "session_id": session_id,
        "device_id": device_id,
        "refresh_token_hash": refresh_hash,
        "is_active": True,
        "active": True
    })

    if not current_session:
        create_audit_log(
            action=AuditAction.LOGOUT_FAILED,
            actor_id=user_id,
            metadata={
                "device_id": device_id,
                "reason": "Invalid session"
            },
            request=request
        )
        return Response({"error": "Invalid session"}, status=403)

    # 🔒 Deactivate ONLY current session
    user_sessions_collection.update_one(
        {"_id": current_session["_id"]},
        {
            "$set": {
                "is_active": False,
                "active": False,
                "logged_out_at": now
            }
        }
    )

    # 🧾 Audit logout
    create_audit_log(
        action=AuditAction.LOGOUT,
        actor_id=user_id,
        metadata={
            "device_id": device_id,
            "session_id": session_id
        },
        request=request
    )

    response = Response(status=200)

    # 🔍 Find another active account on same device
    if ENABLE_ACCOUNT_SWITCHER:
        next_session = user_sessions_collection.find_one(
            {
                "device_id": device_id,
                "is_active": True
            },
            sort=[("last_active", -1)]
        )
    else:
        next_session = None

    if not next_session:
        # 🚪 No account left → full logout
        response.data = {"message": "Logged out"}
        response.delete_cookie("access_token")
        response.delete_cookie("refresh_token")
        return response

    # 🔁 Auto-switch to next account
    next_user_id = str(next_session["user_id"])
    next_session_id = next_session["session_id"]

    new_access = generate_access_token(next_user_id, next_session_id)
    new_refresh = generate_refresh_token(next_user_id, next_session_id)

    user_sessions_collection.update_one(
        {"_id": next_session["_id"]},
        {
            "$set": {
                "active": True,
                "refresh_token_hash": sha256(new_refresh.encode()).hexdigest(),
                "last_active": now
            }
        }
    )

    # 🧾 Audit auto-switch
    create_audit_log(
        action=AuditAction.AUTO_SWITCH_AFTER_LOGOUT,
        actor_id=user_id,
        target_user_id=next_user_id,
        metadata={
            "device_id": device_id
        },
        request=request
    )

    response.data = {
        "message": "Switched to another account",
        "active_user_id": next_user_id
    }

    response.set_cookie(
        key="access_token",
        value=new_access,
        httponly=True,
        secure=env_bool("JWT_COOKIE_SECURE", True),
        samesite=os.getenv("JWT_COOKIE_SAMESITE", "Lax"),
        max_age=int(os.getenv("JWT_ACCESS_TOKEN_EXP", 900))
    )

    response.set_cookie(
        key="refresh_token",
        value=new_refresh,
        httponly=True,
        secure=env_bool("JWT_COOKIE_SECURE", True),
        samesite=os.getenv("JWT_COOKIE_SAMESITE", "Lax"),
        max_age=int(os.getenv("JWT_REFRESH_TOKEN_EXP", 604800))
    )

    return response



@api_view(["GET"])
@jwt_required
def me(request):
    """
    Verify current authenticated user and session
    """

    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")
    ACCESS_COOKIE_NAME = os.getenv("JWT_ACCESS_COOKIE_NAME", "access_token")

    device_id = request.headers.get(DEVICE_ID_HEADER)

    # 1️⃣ Extract access token (cookie → header fallback)
    token = request.COOKIES.get(ACCESS_COOKIE_NAME)

    if not token:
        auth_header = request.headers.get("Authorization")
        if auth_header and auth_header.startswith("Bearer "):
            token = auth_header.split(" ")[1]

    if not token:
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # 2️⃣ Verify JWT
    payload = verify_access_token(token)
    if not payload:
        try:
            create_audit_log(
                action=AuditAction.AUTH_CHECK_FAILED,
                actor_id=None,
                metadata={
                    "reason": "Invalid or expired access token",
                    "device_id": device_id
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    user_id = payload.get("sub")
    session_id = payload.get("session_id")

    if not user_id or not session_id:
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # 3️⃣ Validate session
    session = user_sessions_collection.find_one({
        "user_id": ObjectId(user_id),
        "session_id": session_id,
        "is_active": True
    })

    if not session:
        try:
            create_audit_log(
                action=AuditAction.SESSION_REVOKED,
                actor_id=user_id,
                metadata={
                    "session_id": session_id,
                    "device_id": device_id
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {
                "authenticated": False,
                "reason": "Session revoked"
            },
            status=status.HTTP_401_UNAUTHORIZED
        )

    # 4️⃣ Load user
    user = find_user_by_id(user_id)
    if not user or not user.get("is_active", False):
        try:
            create_audit_log(
                action=AuditAction.AUTH_CHECK_FAILED,
                actor_id=user_id,
                metadata={
                    "reason": "User not found or inactive",
                    "device_id": device_id
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # 5️⃣ Update last activity (silent)
    user_sessions_collection.update_one(
        {"_id": session["_id"]},
        {"$set": {"last_active": datetime.now(timezone.utc)}}
    )

    # 6️⃣ Success audit log
    try:
        create_audit_log(
            action=AuditAction.AUTH_CHECK,
            actor_id=user_id,
            metadata={
                "session_id": session_id,
                "device_id": device_id
            },
            request=request
        )
    except Exception:
        pass

    return Response(
        {
            "authenticated": True,
            "user": serialize_user(user)
        },
        status=status.HTTP_200_OK
    )



@api_view(["POST"])
@jwt_required
def logout_current_device(request):
    """
    Logout user from current device only
    """

    REFRESH_COOKIE_NAME = os.getenv("JWT_REFRESH_COOKIE_NAME", "refresh_token")
    ACCESS_COOKIE_NAME = os.getenv("JWT_ACCESS_COOKIE_NAME", "access_token")
    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")

    device_id = request.headers.get(DEVICE_ID_HEADER)
    refresh_token = request.COOKIES.get(REFRESH_COOKIE_NAME)

    if not refresh_token or not device_id:
        return Response(
            {"error": "Unauthorized"},
            status=status.HTTP_401_UNAUTHORIZED
        )

    refresh_token_hash = hash_refresh_token(refresh_token)

    # 🔒 Deactivate session (safe operation)
    session = user_sessions_collection.find_one_and_update(
        {
            "user_id": ObjectId(request.user_id),
            "refresh_token_hash": refresh_token_hash,
            "device_id": device_id,
            "is_active": True
        },
        {
            "$set": {
                "is_active": False,
                "active": False,
                "logged_out_at": datetime.now(timezone.utc)
            }
        }
    )

    if not session:
        try:
            create_audit_log(
                action=AuditAction.LOGOUT_FAILED,
                actor_id=request.user_id,
                metadata={
                    "device_id": device_id,
                    "reason": "Session not found or already logged out"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Invalid session"},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # ✅ Successful logout audit
    try:
        create_audit_log(
            action=AuditAction.LOGOUT,
            actor_id=request.user_id,
            metadata={
                "device_id": device_id,
                "session_id": session.get("session_id")
            },
            request=request
        )
    except Exception:
        pass

    response = Response(
        {"message": "Logged out from current device"},
        status=status.HTTP_200_OK
    )

    response.delete_cookie(ACCESS_COOKIE_NAME)
    response.delete_cookie(REFRESH_COOKIE_NAME)

    return response


@api_view(["POST"])
@jwt_required
def logout_all_devices(request):
    """
    Logout user from all devices
    """

    ACCESS_COOKIE_NAME = os.getenv("JWT_ACCESS_COOKIE_NAME", "access_token")
    REFRESH_COOKIE_NAME = os.getenv("JWT_REFRESH_COOKIE_NAME", "refresh_token")

    user_id = request.user_id

    # 🔒 Deactivate all active sessions
    result = user_sessions_collection.update_many(
        {
            "user_id": ObjectId(user_id),
            "is_active": True
        },
        {
            "$set": {
                "is_active": False,
                "active": False,
                "logged_out_at": datetime.now(timezone.utc)
            }
        }
    )

    if result.modified_count == 0:
        # ⚠️ No active session found
        try:
            create_audit_log(
                action=AuditAction.LOGOUT_ALL_FAILED,
                actor_id=user_id,
                metadata={
                    "reason": "No active sessions found"
                },
                request=request
            )
        except Exception:
            pass

        response = Response(
            {"message": "No active sessions"},
            status=status.HTTP_200_OK
        )
        response.delete_cookie(ACCESS_COOKIE_NAME)
        response.delete_cookie(REFRESH_COOKIE_NAME)
        return response

    # ✅ Successful logout audit
    try:
        create_audit_log(
            action=AuditAction.LOGOUT_ALL,
            actor_id=user_id,
            metadata={
                "sessions_revoked": result.modified_count
            },
            request=request
        )
    except Exception:
        pass

    response = Response(
        {"message": "Logged out from all devices"},
        status=status.HTTP_200_OK
    )

    response.delete_cookie(ACCESS_COOKIE_NAME)
    response.delete_cookie(REFRESH_COOKIE_NAME)

    return response


@api_view(["POST"])
@jwt_required
def logout_specific_device(request):
    """
    Logout a specific device/session of the user
    """

    ACCESS_COOKIE_NAME = os.getenv("JWT_ACCESS_COOKIE_NAME", "access_token")
    REFRESH_COOKIE_NAME = os.getenv("JWT_REFRESH_COOKIE_NAME", "refresh_token")
    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")

    target_device_id = request.data.get("device_id")
    target_session_id = request.data.get("session_id")

    if not target_device_id and not target_session_id:
        return Response(
            {"error": "device_id or session_id is required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    user_id = request.user_id
    current_device_id = request.headers.get(DEVICE_ID_HEADER)

    # 🔒 Build safe query (ownership enforced)
    query = {
        "user_id": ObjectId(user_id),
        "is_active": True
    }

    if target_device_id:
        query["device_id"] = target_device_id
    if target_session_id:
        query["session_id"] = target_session_id

    session = user_sessions_collection.find_one_and_update(
        query,
        {
            "$set": {
                "is_active": False,
                "active": False,
                "logged_out_at": datetime.now(timezone.utc)
            }
        }
    )

    if not session:
        try:
            create_audit_log(
                action=AuditAction.LOGOUT_DEVICE_FAILED,
                actor_id=user_id,
                metadata={
                    "target_device_id": target_device_id,
                    "target_session_id": target_session_id,
                    "reason": "Session not found or already logged out"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Device not found or already logged out"},
            status=status.HTTP_404_NOT_FOUND
        )

    # ✅ Audit success
    try:
        create_audit_log(
            action=AuditAction.LOGOUT_DEVICE,
            actor_id=user_id,
            metadata={
                "target_device_id": session.get("device_id"),
                "session_id": session.get("session_id"),
                "was_current_device": session.get("device_id") == current_device_id
            },
            request=request
        )
    except Exception:
        pass

    response = Response(
        {"message": "Device logged out successfully"},
        status=status.HTTP_200_OK
    )

    # 🔥 If user logged out CURRENT device → clear cookies
    if session.get("device_id") == current_device_id:
        response.delete_cookie(ACCESS_COOKIE_NAME)
        response.delete_cookie(REFRESH_COOKIE_NAME)

    return response



@api_view(["POST"])
def refresh_token(request):
    """
    Rotate refresh & access token securely
    """

    # ENV CONFIG
    ACCESS_COOKIE_NAME = os.getenv("JWT_ACCESS_COOKIE_NAME", "access_token")
    REFRESH_COOKIE_NAME = os.getenv("JWT_REFRESH_COOKIE_NAME", "refresh_token")
    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")

    ACCESS_EXP = int(os.getenv("JWT_ACCESS_TOKEN_EXP", 900))
    REFRESH_EXP = int(os.getenv("JWT_REFRESH_TOKEN_EXP", 604800))
    COOKIE_SECURE = env_bool("JWT_COOKIE_SECURE", True)
    COOKIE_SAMESITE = os.getenv("JWT_COOKIE_SAMESITE", "None")

    device_id = request.headers.get(DEVICE_ID_HEADER)
    refresh_token_value = request.COOKIES.get(REFRESH_COOKIE_NAME)

    if not refresh_token_value or not device_id:
        return Response({"error": "Unauthorized"}, status=status.HTTP_401_UNAUTHORIZED)

    payload = verify_refresh_token(refresh_token_value)
    if not payload:
        return Response({"error": "Invalid refresh token"}, status=status.HTTP_401_UNAUTHORIZED)

    user_id = payload.get("user_id")
    old_session_id = payload.get("session_id")

    refresh_hash = sha256(refresh_token_value.encode()).hexdigest()

    # Validate session ownership + device binding
    session = user_sessions_collection.find_one({
        "user_id": ObjectId(user_id),
        "session_id": old_session_id,
        "refresh_token_hash": refresh_hash,
        "device_id": device_id,
        "is_active": True
    })

    if not session:
        # Possible token replay / stolen refresh token
        try:
            create_audit_log(
                action=AuditAction.REFRESH_TOKEN_FAILED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "reason": "Session revoked or token reuse detected"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Session revoked"},
            status=status.HTTP_401_UNAUTHORIZED
        )

    # 🔁 ROTATE TOKENS
    new_session_id = str(uuid4())
    new_refresh = generate_refresh_token(user_id, new_session_id)
    new_access = generate_access_token(user_id, new_session_id)

    user_sessions_collection.update_one(
        {"_id": session["_id"]},
        {
            "$set": {
                "session_id": new_session_id,
                "refresh_token_hash": sha256(new_refresh.encode()).hexdigest(),
                "last_active": datetime.now(timezone.utc)
            }
        }
    )

    # ✅ Audit success
    try:
        create_audit_log(
            action=AuditAction.REFRESH_TOKEN,
            actor_id=user_id,
            metadata={
                "device_id": device_id,
                "old_session_id": old_session_id,
                "new_session_id": new_session_id
            },
            request=request
        )
    except Exception:
        pass

    response = Response(
        {"message": "Token refreshed"},
        status=status.HTTP_200_OK
    )

    response.set_cookie(
        key=ACCESS_COOKIE_NAME,
        value=new_access,
        httponly=True,
        secure=COOKIE_SECURE,
        samesite=COOKIE_SAMESITE,
        max_age=ACCESS_EXP
    )

    response.set_cookie(
        key=REFRESH_COOKIE_NAME,
        value=new_refresh,
        httponly=True,
        secure=COOKIE_SECURE,
        samesite=COOKIE_SAMESITE,
        max_age=REFRESH_EXP
    )

    return response


@api_view(["POST"])
def send_otp(request):
    """
    Send OTP for account verification
    """

    # 🌱 ENV CONFIG
    OTP_CHANNEL = os.getenv("OTP_CHANNEL", "sms")  # sms / email
    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")

    ip_address = request.META.get("REMOTE_ADDR")
    ip_address = get_client_ip(request)
    device_id = request.headers.get(DEVICE_ID_HEADER)
    identifier = request.data.get("identifier")

    if not identifier:
        return Response(
            {"error": "Email or phone is required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        validate_identifier(identifier)
        # pass
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )

    # 🔍 Find user (NO ENUMERATION LEAK)
    user = find_user_by_identifier(identifier)
    if not user:
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=None,
                metadata={
                    "identifier": identifier,
                    "device_id": device_id,
                    "reason": "User not found"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"message": "If the account exists, OTP will be sent"},
            status=status.HTTP_200_OK
        )

    if user.get("is_verified", False):
        return Response(
            {"error": "User already verified"},
            status=status.HTTP_400_BAD_REQUEST
        )

    user_id = str(user["_id"])

    # 🚦 Rate limit + abuse protection
    allowed, error = can_send_otp(
        user_id=user_id,
        ip=ip_address,
        identifier=user.get("phone_number")
    )

    if not allowed:
        try:
            create_audit_log(
                action=AuditAction.OTP_RATE_LIMITED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "reason": error
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": error},
            status=status.HTTP_429_TOO_MANY_REQUESTS
        )

    # 🔐 Generate OTP
    otp = generate_otp()
    otp_hash = hash_otp(otp)

    save_otp(
        user_id=user_id,
        otp_hash=otp_hash,
        ip=ip_address,
        to_verify=identifier,
        identifier=identifier
    )

    

    # 📤 Send OTP
    if OTP_CHANNEL == "sms":
 
        send_otp_sms(user["phone_number"], otp, user["country_code"])
    else:
        # send_otp_email(user["primary_email"], otp)
        send_otp_email(
                to_email=user["primary_email"],
                otp=otp,
                username=user.get("first_name"),
                sender_email=DEFAULT_FROM_EMAIL,
                sender_name=os.getenv("SECURITY_SENDER_NAME", "INDZS Security"),
                purpose="Email Recovery Verification"
        )

    # ✅ Audit success

    try:
        create_audit_log(
            action=AuditAction.OTP_SENT,
            actor_id=user_id,
            metadata={
                "device_id": device_id,
                "channel": OTP_CHANNEL,
                "identifier": identifier
            },
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "OTP sent successfully"},
        status=status.HTTP_200_OK
    )




@api_view(["POST"])
def verify_otp(request):
    """
    Verify OTP for account activation
    """

    # 🌱 ENV CONFIG
    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")
    MAX_VERIFY_ATTEMPTS = int(os.getenv("OTP_MAX_VERIFY_ATTEMPTS", 5))

    ip_address = request.META.get("REMOTE_ADDR")
    ip_address = get_client_ip(request)
    device_id = request.headers.get(DEVICE_ID_HEADER)

    identifier = request.data.get("identifier")
    otp_input = request.data.get("otp")

    if not identifier or not otp_input:
        return Response(
            {"error": "Identifier and OTP are required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    try:
        validate_identifier(identifier)
        validate_otp(otp_input)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )

    # 🔍 Find user (ENUMERATION SAFE)
    user = find_user_by_identifier(identifier)
    if not user:
        try:
            create_audit_log(
                action=AuditAction.OTP_VERIFY_FAILED,
                actor_id=None,
                metadata={
                    "identifier": identifier,
                    "device_id": device_id,
                    "reason": "User not found"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Invalid OTP"},
            status=status.HTTP_400_BAD_REQUEST
        )

    user_id = str(user["_id"])

    if user.get("is_verified", False):
        return Response(
            {"error": "User already verified"},
            status=status.HTTP_400_BAD_REQUEST
        )

    otp_record = get_valid_otp(user_id, identifier, identifier)
    if not otp_record:
        try:
            create_audit_log(
                action=AuditAction.OTP_VERIFY_FAILED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "reason": "OTP expired or not found"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "OTP expired or invalid"},
            status=status.HTTP_400_BAD_REQUEST
        )

    # 🚫 Max attempts check
    attempts = otp_record.get("verify_attempts", 0)
    if attempts >= MAX_VERIFY_ATTEMPTS:
        try:
            create_audit_log(
                action=AuditAction.OTP_LOCKED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "attempts": attempts
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Too many wrong attempts"},
            status=status.HTTP_429_TOO_MANY_REQUESTS
        )

    # ❌ Wrong OTP
    if not verify_otp_hash(otp_input, otp_record["otp_hash"]):
        increment_verify_attempt(otp_record["_id"])

        try:
            create_audit_log(
                action=AuditAction.OTP_VERIFY_FAILED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "attempt": attempts + 1,
                    "reason": "Invalid OTP"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Invalid OTP"},
            status=status.HTTP_400_BAD_REQUEST
        )

    # ✅ SUCCESS FLOW
    mark_otp_verified(otp_record["_id"])
    delete_all_user_otps(user_id)
    clear_ip_block(ip_address, user_id)
    mark_user_verified(user_id)

    # ✅ Audit success
    try:
        create_audit_log(
            action=AuditAction.OTP_VERIFIED,
            actor_id=user_id,
            metadata={
                "device_id": device_id,
                "identifier": identifier
            },
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "OTP verified successfully"},
        status=status.HTTP_200_OK
    )


class OtpPurpose:
    LOGIN = "LOGIN"
    EMAIL_VERIFICATION = "EMAIL_VERIFICATION"
    PHONE_VERIFICATION = "PHONE_VERIFICATION"
    RECOVERY_EMAIL = "RECOVERY_EMAIL"
    PASSWORD_RESET = "PASSWORD_RESET"
    ACCOUNT_RECOVERY = "ACCOUNT_RECOVERY"






def purpose_validation_check(user, purpose, identifier):
    
    # 🔒 PURPOSE BASED VALIDATION
    purpose = purpose.upper()    
    if purpose == "PRIMARY_EMAIL":
        if not is_valid_email(identifier):
            return {"is_valid" :False, "error": "Invalid email format", }
        
    if purpose == "RECOVERY_EMAIL":
        if not is_valid_email(identifier):
            return {"is_valid" :False, "error": "Invalid email format", }
            

    elif purpose == "RECOVERY_PHONE":
        if not is_valid_phone(identifier):
            return {"is_valid" :False, "error": "Invalid phone number", }
        
    return {"is_valid" : True}
        
        

def send_recovery_otp(request, purpose):
    """
    Send OTP for recovery actions (email / phone change)
    """
    # 🌱 ENV CONFIG
    DEVICE_ID_HEADER = os.getenv("DEVICE_ID_HEADER", "X-Device-Id")
    DEFAULT_FROM_EMAIL = os.getenv("DEFAULT_FROM_EMAIL", "emailer@noreply.indzs.com")
    SECURITY_SENDER_NAME = os.getenv("SECURITY_SENDER_NAME", "INDZS Security")

    ip_address = request.META.get("REMOTE_ADDR")
    ip_address = get_client_ip(request)
    
    device_id = request.headers.get(DEVICE_ID_HEADER)

    identifier = request.data.get("identifier")
    country_code= request.data.get("country_code")

    user_id = request.user_id

    if not identifier:
        return Response(
            {"error": "Identifier required"},
            status=status.HTTP_400_BAD_REQUEST
        )
    if not country_code:
            return Response(
            {"error": "Country Code required"},
            status=status.HTTP_400_BAD_REQUEST
        )

    # identifier_type=""
    try:
        identifier_type = validate_identifier(identifier)
        validate_country_code(country_code)
    except ValidationError as e:
        return Response(
            {"error": e.message},
            status=status.HTTP_400_BAD_REQUEST
        )
        
    # 🔍 Fetch user
    user = find_user_by_id(user_id)
    if not user or not user.get("is_active", False):
        return Response(
            {"authenticated": False},
            status=status.HTTP_401_UNAUTHORIZED
        )
        
   
    # 🔒 Validate recovery purpose
    is_valid = purpose_validation_check(user, purpose, identifier)
    if not is_valid["is_valid"]:
        try:
            create_audit_log(
                action=AuditAction.RECOVERY_OTP_FAILED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "purpose": purpose,
                    "identifier": identifier,
                    "reason": is_valid["error"]
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": is_valid["error"]},
            status=status.HTTP_400_BAD_REQUEST
        )

    if identifier_type == "phone":
        if(user.get("phone_number")== identifier):
            return Response(
            {"error": "Recovery Phone Number and Primary Phone Number Cannot be same"},
            status=status.HTTP_400_BAD_REQUEST
            )
   

    if identifier_type == "email" and purpose == "recovery_email":
        if(user.get("primary_email") == identifier):
            return Response(
            {"error": "Recovery email and Primary email Cannot be same"},
            status=status.HTTP_400_BAD_REQUEST
            ) 


    # 🚦 Rate limit / abuse protection
    allowed, error = can_send_otp(
        user_id=str(user["_id"]),
        ip=ip_address,
        identifier=identifier,
        purpose=purpose
    )

    if not allowed:
        try:
            create_audit_log(
                action=AuditAction.RECOVERY_OTP_RATE_LIMITED,
                actor_id=user_id,
                metadata={
                    "device_id": device_id,
                    "purpose": purpose,
                    "reason": error
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": error},
            status=status.HTTP_429_TOO_MANY_REQUESTS
        )

    # 🔐 Generate OTP
    otp = generate_otp()
    otp_hash = hash_otp(otp)

    purpose = purpose.lower()
    old_value = user.get(purpose)

    send_to = None
    send_via = None

    # ================= EMAIL RECOVERY =================
    if purpose in ["primary_email", "recovery_email"]:
        if old_value is None or old_value != identifier:
            # 🔁 New email → verify via PHONE
            send_to = user.get("phone_number")
            send_via = "phone"
            send_otp_sms(send_to, otp, user.get("country_code"))
        else:
            # 🔁 Same email → verify via EMAIL
            send_to = identifier
            send_via = "email"
            send_otp_email(
                to_email=send_to,
                otp=otp,
                username=user.get("first_name"),
                sender_email=DEFAULT_FROM_EMAIL,
                sender_name=SECURITY_SENDER_NAME,
                purpose="Email Recovery Verification"
            )

    # ================= PHONE RECOVERY =================
    # elif purpose == "recovery_phone":
    #     send_to = user.get("phone_number") if old_value != identifier else identifier
        
    #     send_via = "phone"
    #     country_code = user.get("country_code") if old_value != identifier else country_code
    #     send_otp_sms(send_to, otp, country_code)
    elif purpose == "recovery_phone":
        print(user.get("recovery_phone_country_code"))
        print(country_code)
        phone_changed = old_value != identifier
        country_changed = user.get("recovery_phone_country_code") != country_code
        print(f"""phone changed {phone_changed}""")
        print(f"""country changed {country_changed}""")
        
        if phone_changed or country_changed:
            send_to = user.get("phone_number")
            send_cc = user.get("country_code")
        else:
            send_to = identifier
            send_cc = country_code


        send_via = "phone"
        send_otp_sms(send_to, otp, send_cc)

    # 💾 Save OTP
    save_otp(
        user_id=str(user["_id"]),
        otp_hash=otp_hash,
        ip=ip_address,
        to_verify=identifier,
        identifier=send_to,
        purpose=purpose,
    )

    # ✅ Audit success
    try:
        create_audit_log(
            action=AuditAction.RECOVERY_OTP_SENT,
            actor_id=user_id,
            metadata={
                "device_id": device_id,
                "purpose": purpose,
                "send_via": send_via,
                "send_to": send_to
            },
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "OTP sent successfully", "send_to" : send_to},
        status=status.HTTP_200_OK
    )



@api_view(["POST"])
@jwt_required
def send_recovery_email_otp(request):
    purpose = request.data.get("purpose")
    if not purpose:
        return Response({"error": "Purpose required"}, status=400)
    return send_recovery_otp(request, purpose)

    


# def update_user_verification(user_id, identifier, purpose, updated_verified=False):
#     """
#     Rules:
#     1. If field value changes -> verification = False
#     2. If field is same and only verification update -> verification = updated_verified
#     3. If old value is None and new value is added -> verification = False
#     """

#     user = users_collection.find_one({"_id": ObjectId(user_id)})
#     if not user:
#         return False

#     update = {}

#     # Map verification field
#     verification_field_map = {
#         "primary_email": "primary_email_verified",
#         "recovery_email": "recovery_email_verified",
#         "recovery_phone": "recovery_phone_verified",
#     }

#     verification_field = verification_field_map.get(purpose)

#     old_value = user.get(purpose)

#     # 🔍 Case 1: New value added (previously null)
#     if old_value is None and identifier is not None:
#         update[purpose] = identifier
#         if verification_field:
#             update[verification_field] = False

#     # 🔄 Case 2: Value changed
#     elif old_value != identifier:
#         update[purpose] = identifier
#         if verification_field:
#             update[verification_field] = False

#     # ✅ Case 3: Value same → only verification update allowed
#     else:
#         if verification_field:
#             update[verification_field] = updated_verified

#     if update:
#         users_collection.update_one(
#             {"_id": ObjectId(user_id)},
#             {"$set": update}
#         )

#     return True




# def update_user_verification(user_id, identifier, purpose, updated_verified=False):
#     """
#     Rules:
#     1. If field value changes -> verification = False
#     2. If field is same and only verification update -> verification = updated_verified
#     3. If old value is None and new value is added -> verification = False
#     """

#     user = users_collection.find_one({"_id": ObjectId(user_id)})
#     if not user:
#         return False

#     # Allowed fields only (prevents privilege escalation)
#     verification_field_map = {
#         "primary_email": "primary_email_verified",
#         "recovery_email": "recovery_email_verified",
#         "recovery_phone": "recovery_phone_verified",
#     }

#     if purpose not in verification_field_map:
#         return False

#     verification_field = verification_field_map[purpose]
#     old_value = user.get(purpose)

#     update = {}

#     # 🆕 Case 1: Old value is None → new value added
#     if old_value is None and identifier:
#         update[purpose] = identifier
#         update[verification_field] = False

#     # 🔄 Case 2: Value changed
#     elif old_value and identifier and old_value != identifier:
#         update[purpose] = identifier
#         update[verification_field] = False

#     # ✅ Case 3: Value same → only verification flag change allowed
#     elif old_value == identifier:
#         update[verification_field] = bool(updated_verified)

#     # 🚫 Disallow clearing identifiers silently
#     else:
#         return False

#     if update:
#         users_collection.update_one(
#             {"_id": ObjectId(user_id)},
#             {"$set": update}
#         )

#     return True



def update_user_verification(
    user_id,
    identifier,
    purpose,
    updated_verified=False,
    country_code=None,   # 🆕 only required for recovery_phone
):
    """
    Rules:
    1. If field value changes -> verification = False
    2. If field is same and only verification update -> verification = updated_verified
    3. If old value is None and new value is added -> verification = False
    4. If recovery_phone OR its country code changes -> verification = False
    """

    user = users_collection.find_one({"_id": ObjectId(user_id)})
    if not user:
        return False

    verification_field_map = {
        "primary_email": "primary_email_verified",
        "recovery_email": "recovery_email_verified",
        "recovery_phone": "recovery_phone_verified",
    }

    if purpose not in verification_field_map:
        return False

    verification_field = verification_field_map[purpose]
    update = {}

    # =========================
    # 📞 Recovery phone logic
    # =========================
    if purpose == "recovery_phone":
        old_phone = user.get("recovery_phone")
        old_cc = user.get("recovery_phone_country_code")

        if not identifier or not country_code:
            return False

        phone_changed = old_phone != identifier
        cc_changed = old_cc != country_code

        # 🆕 New phone added
        if old_phone is None:
            update["recovery_phone"] = identifier
            update["recovery_phone_country_code"] = country_code
            update[verification_field] = False

        # 🔄 Phone or country code changed
        elif phone_changed or cc_changed:
            update["recovery_phone"] = identifier
            update["recovery_phone_country_code"] = country_code
            update[verification_field] = False

        # ✅ Same phone + same country code
        else:
            update[verification_field] = bool(updated_verified)

    # =========================
    # 📧 Email logic (unchanged)
    # =========================
    else:
        old_value = user.get(purpose)

        # 🆕 New value added
        if old_value is None and identifier:
            update[purpose] = identifier
            update[verification_field] = False

        # 🔄 Value changed
        elif old_value and identifier and old_value != identifier:
            update[purpose] = identifier
            update[verification_field] = False

        # ✅ Same value → verification update
        elif old_value == identifier:
            update[verification_field] = bool(updated_verified)

        else:
            return False

    if update:
        users_collection.update_one(
            {"_id": ObjectId(user_id)},
            {"$set": update}
        )

    return True



@api_view(["POST"])
@jwt_required
def verify_recovery_otp(request):
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    
    identifier = request.data.get("identifier")     # where OTP was sent
    otp_input = request.data.get("otp")
    purpose = request.data.get("purpose")
    to_verify = request.data.get("to_verify")       # value being verified
    country_code = request.data.get("country_code")

    
    user_id = request.user_id

    MAX_ATTEMPTS = int(os.getenv("MAX_VERIFY_ATTEMPTS", 5))

    # 1️⃣ Basic validation
    if not all([identifier, otp_input, purpose, to_verify, country_code]):
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=user_id,
                metadata={"reason": "missing_fields"},
                request=request
            )
        except Exception:
            pass

        return Response({"error": "Missing fields"}, status=400)

    purpose = purpose.lower()


    try:
        validate_identifier(identifier)
        validate_identifier(to_verify)
        validate_otp(otp_input)
        validate_country_code(country_code)
    except ValidationError as e:
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=user_id,
                metadata={
                    "purpose": purpose,
                    "reason": "validation_failed",
                    "error": e.message
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": e.message}, status=400)

    # 2️⃣ Fetch user
    user = find_user_by_id(user_id)
    if not user or not user.get("is_active", False):
        try:
            create_audit_log(
                action=AuditAction.AUTH_CHECK_FAILED,
                actor_id=user_id,
                metadata={"reason": "user_inactive_or_missing"},
                request=request
            )
        except Exception:
            pass

        return Response({"authenticated": False}, status=401)

    # 3️⃣ Fetch OTP
    otp_record = get_valid_otp(
        user_id=str(user["_id"]),
        to_verify=to_verify,
        identifier=identifier,
        purpose=purpose
    )

    if not otp_record:
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=str(user["_id"]),
                metadata={
                    "purpose": purpose,
                    "reason": "otp_expired_or_invalid"
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "OTP expired or invalid"}, status=400)

    # 4️⃣ Attempt limit
    attempts = otp_record.get("verify_attempts", 0)
    if attempts >= MAX_ATTEMPTS:
        try:
            create_audit_log(
                action=AuditAction.OTP_RATE_LIMITED,
                actor_id=str(user["_id"]),
                metadata={
                    "purpose": purpose,
                    "attempts": attempts
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "Too many attempts"}, status=429)

    # 5️⃣ Verify OTP
    if not verify_otp_hash(otp_input, otp_record["otp_hash"]):
        increment_verify_attempt(otp_record["_id"])

        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=str(user["_id"]),
                metadata={
                    "purpose": purpose,
                    "attempts": attempts + 1,
                    "reason": "invalid_otp"
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "Invalid OTP"}, status=400)

    # ✅ 6️⃣ SUCCESS FLOW
    mark_otp_verified(otp_record["_id"])

    update_user_verification(
        user_id=user["_id"],
        identifier=to_verify,
        purpose=purpose,
        updated_verified=True,
        country_code=country_code
    )
    users_collection.update_one(
        {"_id": user["_id"]},
        {"$set": {
            "recovery_phone_country_code": country_code 
        }}
    )

    delete_all_user_otps(str(user["_id"]), purpose)
    clear_ip_block(ip, str(user["_id"]))

    try:
        create_audit_log(
            action=AuditAction.SETTINGS_UPDATED,
            actor_id=str(user["_id"]),
            metadata={
                "purpose": purpose,
                "verified": True
            },
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "OTP verified successfully"},
        status=status.HTTP_200_OK
    )



@api_view(["POST"])
def send_forgot_password_otp(request):
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    
    identifier = request.data.get("identifier")  # email or phone

    OTP_PURPOSE = OtpPurpose.PASSWORD_RESET

    # 🔒 ENV configs
    SENDER_NAME = os.getenv("SECURITY_SENDER_NAME", "INDZS Security")
    DEFAULT_EMAIL = os.getenv("DEFAULT_FROM_EMAIL", "emailer@noreply.indzs.com")

    # 1️⃣ Basic validation
    if not identifier:
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=None,
                metadata={
                    "purpose": OTP_PURPOSE,
                    "reason": "identifier_missing"
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "Email or phone required"}, status=400)

    try:
        validate_identifier(identifier)
    except ValidationError as e:
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=None,
                metadata={
                    "purpose": OTP_PURPOSE,
                    "reason": "invalid_identifier",
                    "identifier": identifier
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": e.message}, status=status.HTTP_400_BAD_REQUEST)

    # 2️⃣ Find user
    user = find_user_by_any_identifier(identifier)
    if not user or not user.get("is_active", False):
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=None,
                metadata={
                    "purpose": OTP_PURPOSE,
                    "reason": "user_not_found_or_inactive",
                    "identifier": identifier
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "User not found"}, status=404)

    user_id = str(user["_id"])

    # 3️⃣ Rate limit check
    allowed, error = can_send_otp(
        user_id,
        ip,
        identifier=identifier,
        purpose=OTP_PURPOSE
    )

    if not allowed:
        try:
            create_audit_log(
                action=AuditAction.OTP_RATE_LIMITED,
                actor_id=user_id,
                metadata={
                    "purpose": OTP_PURPOSE,
                    "identifier": identifier,
                    "reason": error
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": error}, status=429)

    # 4️⃣ Generate OTP
    otp = generate_otp()
    otp_hash = hash_otp(otp)

    save_otp(
        user_id=user_id,
        otp_hash=otp_hash,
        ip=ip,
        purpose=OTP_PURPOSE,
        identifier=identifier,
        to_verify=identifier,
    )

    # 5️⃣ Send OTP
    try:
        if "@" in identifier:
            send_otp_email(
                to_email=identifier,
                otp=otp,
                username=user.get("first_name", "User"),
                sender_email=DEFAULT_EMAIL,
                sender_name=SENDER_NAME,
                purpose="Password Reset"
            )
        else:
            country_code = "91"
            if user.get("phone_number") == identifier:
                 country_code= user.get("country_code")
            
            send_otp_sms(
                identifier,
                otp,
                country_code,
                purpose=OTP_PURPOSE
            )
    except Exception:
        # ❗ OTP generated but delivery failed
        try:
            create_audit_log(
                action=AuditAction.OTP_SEND_FAILED,
                actor_id=user_id,
                metadata={
                    "purpose": OTP_PURPOSE,
                    "identifier": identifier,
                    "reason": "delivery_failed"
                },
                request=request
            )
        except Exception:
            pass

        return Response(
            {"error": "Failed to send OTP. Please try again."},
            status=500
        )

    # ✅ 6️⃣ Success audit
    try:
        create_audit_log(
            action=AuditAction.OTP_SENT,
            actor_id=user_id,
            metadata={
                "purpose": OTP_PURPOSE,
                "identifier": identifier,
                "channel": "email" if "@" in identifier else "sms"
            },
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "OTP sent for password reset"},
        status=200
    )



@api_view(["POST"])
def verify_forgot_password_otp(request):
    identifier = request.data.get("identifier")
    otp_input = request.data.get("otp")

    MAX_ATTEMPTS = int(os.getenv("MAX_VERIFY_ATTEMPTS", 5))
    PURPOSE = OtpPurpose.PASSWORD_RESET

    if not identifier or not otp_input:
        return Response({"error": "Identifier and OTP required"}, status=400)

    try:
        validate_identifier(identifier)
        validate_otp(otp_input)
    except ValidationError as e:
        return Response({"error": e.message}, status=400)

    user = find_user_by_any_identifier(identifier)
    if not user:
        return Response({"error": "Invalid OTP"}, status=400)

    user_id = str(user["_id"])

    otp_record = get_valid_otp(
        user_id=user_id,
        to_verify=identifier,
        identifier=identifier,
        purpose=PURPOSE
    )

    if not otp_record:
        try:
            create_audit_log(
                action=AuditAction.OTP_VERIFY_FAILED,
                actor_id=user_id,
                metadata={
                    "purpose": PURPOSE,
                    "reason": "expired_or_invalid"
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "OTP expired or invalid"}, status=400)

    attempts = otp_record.get("verify_attempts", 0)
    if attempts >= MAX_ATTEMPTS:
        try:
            create_audit_log(
                action=AuditAction.OTP_VERIFY_BLOCKED,
                actor_id=user_id,
                metadata={"purpose": PURPOSE},
                request=request
            )
        except Exception:
            pass

        return Response({"error": "Too many attempts"}, status=429)

    if not verify_otp_hash(otp_input, otp_record["otp_hash"]):
        increment_verify_attempt(otp_record["_id"])

        try:
            create_audit_log(
                action=AuditAction.OTP_VERIFY_FAILED,
                actor_id=user_id,
                metadata={
                    "purpose": PURPOSE,
                    "attempts": attempts + 1
                },
                request=request
            )
        except Exception:
            pass

        return Response({"error": "Invalid OTP"}, status=400)

    # ✅ SUCCESS
    mark_otp_verified(otp_record["_id"])

    try:
        create_audit_log(
            action=AuditAction.OTP_VERIFIED,
            actor_id=user_id,
            metadata={"purpose": PURPOSE},
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "OTP verified. You may now reset password"},
        status=200
    )


from apps.accounts.views import update_password_all_apps

@api_view(["POST"])
def reset_password(request):
    identifier = request.data.get("identifier")
    new_password = request.data.get("new_password")

    BCRYPT_ROUNDS = int(os.getenv("BCRYPT_ROUNDS", 12))
    PURPOSE = OtpPurpose.PASSWORD_RESET

    if not identifier or not new_password:
        return Response({"error": "Missing fields"}, status=400)

    try:
        validate_identifier(identifier)
        validate_password(new_password)
    except ValidationError as e:
        return Response({"error": e.message}, status=400)

    user = find_user_by_any_identifier(identifier)
    if not user:
        return Response({"error": "Invalid request"}, status=400)

    user_id = str(user["_id"])

    otp_record = otp_collection.find_one({
        "user_id": user["_id"],
        "purpose": PURPOSE,
        "verified": True,
        "expires_at": {"$gte": datetime.now(timezone.utc)}
    })

    if not otp_record:
        return Response(
            {"error": "OTP verification required"},
            status=403
        )

    # 🔐 Update passwords
    hashed_password = hash_password(new_password)
    hashed_chitraplay_password = bcrypt_password_hash(new_password)
    hashed_kujanam_password = custom_bcrypt_password_hash(
        new_password,
        BCRYPT_ROUNDS
    )

    users_collection.update_one(
        {"_id": user["_id"]},
        {"$set": {
            "password": hashed_password,
            "chitraplay_password": hashed_chitraplay_password,
            "bhulok_password": hashed_chitraplay_password,
            "kujanam_password": hashed_kujanam_password
        }}
    )

    update_password_all_apps(user, new_password)

    # 🧹 Cleanup OTPs
    otp_collection.delete_many({
        "user_id": user["_id"],
        "purpose": PURPOSE
    })

    # 🔥 Kill all sessions
    user_sessions_collection.update_many(
        {"user_id": user["_id"]},
        {"$set": {"is_active": False}}
    )

    # 🧾 AUDIT LOGS (SAFE)
    try:
        create_audit_log(
            action=AuditAction.PASSWORD_RESET,
            actor_id=user_id,
            metadata={
                "sessions_revoked": True
            },
            request=request
        )
    except Exception:
        pass

    try:
        create_audit_log(
            action=AuditAction.SESSION_REVOKED,
            actor_id=user_id,
            metadata={"reason": "password_reset"},
            request=request
        )
    except Exception:
        pass

    return Response(
        {"message": "Password updated successfully"},
        status=200
    )

def mask_email(email):
    if not email or "@" not in email:
        return None
    name, domain = email.split("@")
    return name[:2] + "*" * max(len(name) - 2, 1) + "@" + domain


def mask_phone(phone):
    if not phone or len(phone) < 4:
        return None
    return "*" * (len(phone) - 4) + phone[-4:]


@api_view(["POST"])
def try_another_way(request):
    identifier = request.data.get("identifier")

    if not identifier:
        return Response(
            {"error": "identifier required"},
            status=400
        )

    # 🔍 Find user by any known identifier
    user = users_collection.find_one({
        "$or": [
            {"primary_email": identifier},
            {"phone_number": identifier},
            {"recovery_email": identifier},
            {"recovery_phone": identifier},
        ]
    })

    # 🔐 Always return safely
    if not user:
        return Response({"methods": []}, status=200)

    methods = []

    # 📱 Phone number
    if user.get("phone_number"):
        methods.append({
            "type": "PHONE",
            "label": "SMS",
            "value": mask_phone(user["phone_number"])
        })
        
    # 📧 Primary email
    if user.get("primary_email"):
        methods.append({
            "type": "PRIMARY_EMAIL",
            "label": "Email",
            "value": mask_email(user["primary_email"])
        })

    # 📱 Recovery phone
    if user.get("recovery_phone"):
        methods.append({
            "type": "RECOVERY_PHONE",
            "label": "Recovery Phone",
            "value": mask_phone(user["recovery_phone"])
        })
        
    # 📧 Recovery email
    if user.get("recovery_email"):
        methods.append({
            "type": "RECOVERY_EMAIL",
            "label": "Recovery Email",
            "value": mask_email(user["recovery_email"])
        })



    return Response(
        {
            "methods": methods
        },
        status=200
    )



import uuid
from datetime import timedelta, timezone

from core.db.mongo import account_recovery_otp_collection
from .models.otp import can_send_account_otp,save_account_otp
from datetime import datetime, timezone

def build_find_account_query(first_name, identifier, last_name=None):
    query = {
        "first_name": {"$regex": f"^{first_name}$", "$options": "i"},
        "$or": [
            {"phone_number": identifier},
            {"primary_email": identifier},
            {"recovery_phone": identifier},
            {"recovery_email": identifier},
        ]
    }
    if last_name:
        query["last_name"] = {"$regex": f"^{last_name}$", "$options": "i"}
    return query




@api_view(["POST"])
def find_account(request):
    first_name = request.data.get("first_name")
    last_name = request.data.get("last_name")
    identifier = request.data.get("identifier")
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    
    DEFAULT_EMAIL = os.getenv("DEFAULT_FROM_EMAIL", "emailer@noreply.indzs.com")
    if not first_name or not identifier:
        return Response({"error": "first_name and identifier required"}, status=400)

    try:
        validate_identifier(identifier)
        validate_name(first_name)
        if last_name:
            validate_name(last_name)
    except ValidationError as e:
        return Response({"error": e.message}, status=400)

    users = list(users_collection.find(
        build_find_account_query(first_name, identifier, last_name)
    ))

    if not users:
        return Response({"error": "No accounts found"}, status=404)

    allowed, error = can_send_account_otp(
        ip=ip,
        identifier=identifier,
        purpose="ACCOUNT_RECOVERY"
    )
    if not allowed:
        return Response({"error": error}, status=429)

    otp = generate_otp()
    otp_hash = hash_otp(otp)
    recovery_token = str(uuid.uuid4())

    account_recovery_otp_collection.insert_one({
        "recovery_token": recovery_token,
        "identifier": identifier,
        "otp_hash": otp_hash,
        "matched_user_ids": [str(u["_id"]) for u in users],
        "attempts": 1,
        "ip_address": ip,
        "created_at": datetime.now(timezone.utc),
        "expires_at": datetime.now(timezone.utc) + timedelta(
            minutes=int(os.getenv("ACCOUNT_OTP_EXPIRY_MINUTES", 10))
        ),
        "verified": False
    })

    try:
        if "@" in identifier:
            send_otp_email(
                to_email=identifier,
                otp=otp,
                username="User",
                sender_email=DEFAULT_EMAIL,
                sender_name=os.getenv("SECURITY_SENDER_NAME", "INDZS Security"),
                purpose="Account Recovery"
            )
        else:
            send_otp_sms(identifier, otp, purpose="Account Recovery")
    finally:
        safe_audit(
            action=AuditAction.ACCOUNT_RECOVERY_OTP_SENT,
            actor_id=str(identifier),
            metadata={
                "ip":ip,
                "purpose": "account_recovery",
            },
            request=request
        )


    return Response({"message": "OTP sent", "recovery_token": recovery_token}, status=200)




@api_view(["POST"])
def verify_find_account_otp(request):
    recovery_token = request.data.get("recovery_token")
    otp = request.data.get("otp")
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    

    if not recovery_token or not otp:
        return Response({"error": "recovery_token and otp required"}, status=400)

    try:
        validate_otp(otp)
    except ValidationError as e:
        return Response({"error": e.message}, status=400)

    record = account_recovery_otp_collection.find_one({
        "recovery_token": recovery_token,
        "verified": False
    })

    if not record:
        return Response({"error": "Invalid or used OTP"}, status=400)

    expires_at = record["expires_at"]
    if expires_at.tzinfo is None:
        expires_at = expires_at.replace(tzinfo=timezone.utc)

    if expires_at < datetime.now(timezone.utc):
        return Response({"error": "OTP expired"}, status=400)

    if not verify_otp_hash(otp, record["otp_hash"]):
        safe_audit(
            action=AuditAction.ACCOUNT_RECOVERY_OTP_FAILED,
            actor_id=str(record["identifier"]),
            metadata={
                "purpose": "account_recovery",
                "ip": ip
            },
            request=request
        )

        return Response({"error": "Invalid OTP"}, status=400)

    account_recovery_otp_collection.update_one(
        {"_id": record["_id"]},
        {"$set": {"verified": True}}
    )

    safe_audit(
            action=AuditAction.ACCOUNT_RECOVERY_OTP_FAILED,
            actor_id=str(record["identifier"]),
            metadata={
                "purpose": "account_recovery",
                "ip": ip
            },
            request=request
    )

    users = list(users_collection.find(
        {"_id": {"$in": [ObjectId(i) for i in record["matched_user_ids"]]}},
        {"password": 0}
    ))

    return Response({
        "accounts": [{
            "user_id": str(u["_id"]),
            "name": f"{u.get('first_name')} {u.get('last_name', '')}".strip(),
            "masked_email": u.get("primary_email"),
            "masked_phone": u.get("phone_number")
        } for u in users]
    }, status=200)



@api_view(["POST"])
@jwt_required
@parser_classes([MultiPartParser, FormParser])
def upload_profile_picture(request):
    file = request.FILES.get("profile_picture")
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    
    user_id = str(request.user_id)

    if not file:
        return Response({"error": "profile_picture is required"}, status=400)

    try:
        processed_image = process_profile_image(file)
    except ValueError as e:
        return Response({"error": str(e)}, status=400)

    media_cdn_base = os.getenv("MEDIA_CDN_BASE_URL","")
    if not media_cdn_base:
        return Response({"error": "Media configuration error"}, status=500)

    storage_path = f"profile_pictures/user_{user_id}.jpg"
    cdn_path = f"{media_cdn_base}/{storage_path}"

    if default_storage.exists(storage_path):
        default_storage.delete(storage_path)

    default_storage.save(storage_path, processed_image)

    users_collection.update_one(
        {"_id": ObjectId(user_id)},
        {"$set": {"profile_picture": cdn_path}}
    )

    safe_audit(
        action=AuditAction.SETTINGS_UPDATED,
        actor_id=str(request.user_id),
        metadata={"field": "profile_picture"},
        request=request
    )

    return Response(
        {
            "message": "Profile picture updated",
            "profile_picture": cdn_path
        },
        status=200
    )




@api_view(["DELETE"])
@jwt_required
def remove_profile_picture(request):
    user_id = str(request.user_id)
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    

    user = users_collection.find_one(
        {"_id": ObjectId(user_id)},
        {"profile_picture": 1}
    )

    if user and user.get("profile_picture"):
        media_cdn_base = os.getenv("MEDIA_CDN_BASE_URL", "")
        storage_path = user["profile_picture"].replace(
            f"{media_cdn_base}/", ""
        )

        if default_storage.exists(storage_path):
            default_storage.delete(storage_path)

        users_collection.update_one(
            {"_id": ObjectId(user_id)},
            {"$unset": {"profile_picture": ""}}
        )

        safe_audit(
            action=AuditAction.SETTINGS_UPDATED,
            actor_id=str(request.user_id),
            metadata={
                "field": "profile_picture",
                "action": "removed"
            },
            request=request
        )


    return Response(
        {"message": "Profile picture removed"},
        status=200
    )




@api_view(["POST"])
@jwt_required
def get_logged_in_devices(request):
    user_id = str(request.user_id)
    ip = request.META.get("REMOTE_ADDR")
    ip = get_client_ip(request)
    

    sessions = list(
        user_sessions_collection.find(
            {
                "user_id": ObjectId(user_id),
                "is_active": True,
            },
            {
                "_id": 0,
                "session_id": 1,
                "device_id": 1,
                "browser": 1,
                "os": 1,
                "ip_address": 1,
                "created_at": 1,
                "last_active": 1,
                "is_active": 1,
                "logged_out_at": 1,
                "country_name" : 1,
                "city" : 1,
                "region" : 1,
                "country" : 1
                
            }
        ).sort("created_at", -1)
    )

    total_devices = len(sessions)
    active_devices = sum(1 for s in sessions if s.get("is_active"))

    safe_audit(
        action=AuditAction.VIEW_LOGGED_IN_ACCOUNTS,
        actor_id=str(request.user_id),
        metadata={
            "device_count": total_devices
        },
        request=request
    )


    return Response(
        {
            "total_devices": total_devices,
            "active_devices": active_devices,
            "devices": sessions,
        },
        status=status.HTTP_200_OK
    )
