from rest_framework.decorators import api_view
from rest_framework.response import Response
from rest_framework import status
from datetime import datetime, timezone
from bson import ObjectId
from django.core.exceptions import ValidationError
from core.utils.helpers import generate_ticket_number, get_client_ip
from core.utils.validation import validate_ticket_payload
from core.db.mongo import support_tickets_collection, users_collection
from .decorators import jwt_optional

def extract_user_context(request):
    if not request.user_id:
        return {
            "is_authenticated": False,
            "user_id": None,
            "user_email": None,
            "user_phone": None
        }

    user = users_collection.find_one(
        {"_id": ObjectId(request.user_id), "is_deleted": {"$ne": True}},
        {
            "primary_email": 1,
            "phone_number": 1
        }
    )

    if not user:
        return {
            "is_authenticated": False,
            "user_id": None,
            "user_email": None,
            "user_phone": None
        }

    return {
        "is_authenticated": True,
        "user_id": ObjectId(request.user_id),
        "user_email": user.get("primary_email"),
        "user_phone": user.get("phone_number")
    }


@api_view(["POST"])
@jwt_optional
def create_support_ticket(request):
    data = request.data

    try:
        validate_ticket_payload(data)
    except ValidationError as e:
        return Response(
            {"error": e},
            status=status.HTTP_400_BAD_REQUEST
        )
    ticket_number = generate_ticket_number()

    # 🌐 Request metadata
    ip_address = get_client_ip(request)
    device_id = request.headers.get("X-Device-Id", "")  # optional
    user_agent = request.META.get("HTTP_USER_AGENT", "")

    # 👤 Optional auth context
    user_context = extract_user_context(request)

    ticket = {
        "ticket_number": ticket_number,

        # User submitted data
        "name": data["name"].strip(),
        "email": data["email"].lower().strip(),
        "phone": data["phone"].strip(),
        "category": data["category"],
        "subject": data["subject"].strip(),
        "description": data["description"].strip(),

        # System fields
        "status": "OPEN",
        "created_at": datetime.now(timezone.utc),
        "updated_at": datetime.now(timezone.utc),
        "resolved_at": None,

        # Device / network
        "ip_address": ip_address,
        "device_id": device_id,
        "user_agent": user_agent,

        # Optional user context
        "is_authenticated": user_context["is_authenticated"],
        "user_id": user_context["user_id"],
        "user_email": user_context["user_email"],
        "user_phone": user_context["user_phone"],
    }

    support_tickets_collection.insert_one(ticket)

    return Response(
        {
            "message": "Your query has been submitted successfully",
            "ticket_number": ticket_number
        },
        status=status.HTTP_201_CREATED
    )


# @api_view(["POST"])
# def create_support_ticket(request):
#     data = request.data
#     validate_ticket_payload(data)

#     ticket_number = generate_ticket_number()

#     ip_address = get_client_ip(request)
#     device_id = request.headers.get("X-Device-Id")  # optional
#     user_agent = request.META.get("HTTP_USER_AGENT", "")

#     user_context = extract_user_context(request)

#     ticket = {
#         "ticket_number": ticket_number,

#         # User provided data
#         "name": data["name"].strip(),
#         "email": data["email"].lower().strip(),
#         "phone": data["phone"].strip(),
#         "category": data["category"],
#         "subject": data["subject"].strip(),
#         "description": data["description"].strip(),

#         # System metadata
#         "status": "OPEN",
#         "created_at": datetime.now(timezone.utc),
#         "updated_at": datetime.now(timezone.utc),
#         "resolved_at": None,

#         # Optional device info
#         "ip_address": ip_address,
#         "device_id": device_id,
#         "user_agent": user_agent,

#         # Auth context (nullable)
#         "user_id": user_context["user_id"],
#         "user_email": user_context["user_email"],
#         "user_phone": user_context["user_phone"],
#         "is_authenticated": user_context["is_authenticated"],
#     }

#     support_tickets_collection.insert_one(ticket)

#     return Response(
#         {
#             "message": "Your query has been submitted successfully",
#             "ticket_number": ticket_number
#         },
#         status=status.HTTP_201_CREATED
#     )

