import os
class AuditAction:
    ALL = "ALL"
    LOGIN = "LOGIN"
    LOGIN_FAILED = "LOGIN_FAILED"
    LOGOUT = "LOGOUT"
    FORCE_LOGOUT = "FORCE_LOGOUT"
    PASSWORD_RESET = "PASSWORD_RESET"
    PASSWORD_CHANGE = "PASSWORD_CHANGE"
    EMAIL_VERIFIED = "EMAIL_VERIFIED"
    PHONE_VERIFIED = "PHONE_VERIFIED"
    USER_BLOCKED = "USER_BLOCKED"
    USER_UNBLOCKED = "USER_UNBLOCKED"
    ROLE_CHANGED = "ROLE_CHANGED"
    APP_CREATED = "APP_CREATED"
    APP_UPDATED = "APP_UPDATED"
    SETTINGS_UPDATED = "SETTINGS_UPDATED"
    SWITCH_ACCOUNT = "SWITCH_ACCOUNT"
    USER_REGISTERED = "USER_REGISTERED"
    VIEW_LOGGED_IN_ACCOUNTS = "VIEW_LOGGED_IN_ACCOUNTS"
    SWITCH_ACCOUNT_FAILED = "SWITCH_ACCOUNT_FAILED"
    DELETE_ACCOUNT_FROM_DEVICE = "DELETE_ACCOUNT_FROM_DEVICE"
    DELETE_ACCOUNT_FROM_DEVICE_FAILED = "DELETE_ACCOUNT_FROM_DEVICE_FAILED"
    LOGOUT_FAILED = "LOGOUT_FAILED"
    AUTO_SWITCH_AFTER_LOGOUT = "AUTO_SWITCH_AFTER_LOGOUT"
    AUTH_CHECK = "AUTH_CHECK"
    AUTH_CHECK_FAILED = "AUTH_CHECK_FAILED"
    SESSION_REVOKED = "SESSION_REVOKED"
    LOGOUT_ALL = "LOGOUT_ALL"
    LOGOUT_ALL_FAILED = "LOGOUT_ALL_FAILED"
    REFRESH_TOKEN = "REFRESH_TOKEN"
    REFRESH_TOKEN_FAILED = "REFRESH_TOKEN_FAILED"
    OTP_SENT = "OTP_SENT"
    OTP_SEND_FAILED = "OTP_SEND_FAILED"
    OTP_RATE_LIMITED = "OTP_RATE_LIMITED"
    ACCOUNT_RECOVERY_OTP_FAILED="ACCOUNT_RECOVERY_OTP_FAILED"
    ACCOUNT_RECOVERY_OTP_SENT="ACCOUNT_RECOVERY_OTP_SENT"




from datetime import datetime
from bson import ObjectId
from typing import Optional, Dict, Any

from core.db.mongo import audit_logs




def create_audit_log(
    *,
    action: str,
    actor_id: Optional[str],
    target_user_id: Optional[str] = None,
    target_app_id: Optional[str] = None,
    metadata: Optional[Dict[str, Any]] = None,
    request=None
):
    """
    Central audit log creator
    """
    
    if not os.getenv("AUDIT_LOG_ENABLED", True):
        return
    
    try:
        log = {
            "action": action,
            "actor_id": ObjectId(actor_id) if actor_id else None,
            "target_user_id": ObjectId(target_user_id) if target_user_id else None,
            "target_app_id": ObjectId(target_app_id) if target_app_id else None,
            "metadata": metadata or {},
            "created_at": datetime.utcnow()
        }

        if request:
            log.update({
                "ip_address": request.META.get("REMOTE_ADDR"),
                "user_agent": request.META.get("HTTP_USER_AGENT")
            })

        audit_logs.insert_one(log)
    except Exception:
        pass






def get_audit_logs(
    *,
    action: Optional[str] = None,
    actor_id: Optional[str] = None,
    target_user_id: Optional[str] = None,
    target_app_id: Optional[str] = None,
    date_from=None,
    date_to=None,
    page: int = 1,
    limit: int = 20
):
    query = {}

    if action:
        query["action"] = action

    if actor_id:
        query["actor_id"] = ObjectId(actor_id)

    if target_user_id:
        query["target_user_id"] = ObjectId(target_user_id)

    if target_app_id:
        query["target_app_id"] = ObjectId(target_app_id)

    if date_from or date_to:
        query["created_at"] = {}
        if date_from:
            query["created_at"]["$gte"] = date_from
        if date_to:
            query["created_at"]["$lte"] = date_to

    skip = (page - 1) * limit

    cursor = (
        audit_logs
        .find(query)
        .sort("created_at", -1)
        .skip(skip)
        .limit(limit)
    )

    results = []
    for log in cursor:
        log["_id"] = str(log["_id"])
        if log.get("actor_id"):
            log["actor_id"] = str(log["actor_id"])
        if log.get("target_user_id"):
            log["target_user_id"] = str(log["target_user_id"])
        if log.get("target_app_id"):
            log["target_app_id"] = str(log["target_app_id"])
        results.append(log)

    total = audit_logs.count_documents(query)

    return {
        "data": results,
        "page": page,
        "limit": limit,
        "total": total
    }




