import ipaddress
import geoip2.database
from django.conf import settings

# Load databases once (IMPORTANT for performance)
CITY_READER = geoip2.database.Reader(
    settings.GEOIP_CITY_DB_PATH
)
ASN_READER = geoip2.database.Reader(
    settings.GEOIP_ASN_DB_PATH
)

# -------------------------------
# IP Utilities
# -------------------------------

def is_private_ip(ip: str) -> bool:
    try:
        return ipaddress.ip_address(ip).is_private
    except Exception:
        return True


def extract_client_ip(request) -> str | None:
    """
    Extract real client IP behind proxies / Cloudflare
    """
    # Cloudflare
    cf_ip = request.headers.get("CF-Connecting-IP")
    if cf_ip:
        return cf_ip.strip()

    # Standard proxy headers
    xff = request.headers.get("X-Forwarded-For")
    if xff:
        return xff.split(",")[0].strip()

    # Fallback
    return request.META.get("REMOTE_ADDR")


# -------------------------------
# Geo Lookup
# -------------------------------

def geo_lookup(ip: str) -> dict | None:
    """
    Offline IP → Geo lookup using MaxMind
    """
    try:
        city = CITY_READER.city(ip)
        asn = ASN_READER.asn(ip)

        return {
            "country": city.country.iso_code,
            "country_name": city.country.name,
            "city": city.city.name,
            "region": city.subdivisions.most_specific.name,
            "lat": city.location.latitude,
            "lng": city.location.longitude,
            "timezone": city.location.time_zone,
            "asn": asn.autonomous_system_number,
            "org": asn.autonomous_system_organization
        }

    except Exception:
        return None


# -------------------------------
# Public API (USE THIS)
# -------------------------------

def get_geo_context(request) -> dict:
    """
    Production-ready Geo Context extractor
    """

    ip = extract_client_ip(request)

    # Absolute fallback
    if not ip:
        return _dev_geo("UNKNOWN")

    # Localhost / Private IP handling
    if is_private_ip(ip):
        if settings.DEBUG:
            return _dev_geo("LOCALHOST")
        return {
            "ip": ip,
            "country": None,
            "city": None,
            "lat": None,
            "lng": None,
            "timezone": None,
            "asn": None,
            "is_private": True
        }

    geo = geo_lookup(ip)

    if not geo:
        return {
            "ip": ip,
            "country": None,
            "city": None,
            "lat": None,
            "lng": None,
            "timezone": None,
            "asn": None,
            "is_private": False
        }

    return {
        "ip": ip,
        **geo,
        "is_private": False
    }


# -------------------------------
# DEV Helpers
# -------------------------------

def _dev_geo(label: str) -> dict:
    """
    Fake geo for local testing
    """
    return {
        "ip": "127.0.0.1",
        "country": "DEV",
        "country_name": label,
        "city": label,
        "region": label,
        "lat": 0.0,
        "lng": 0.0,
        "timezone": "UTC",
        "asn": 0,
        "org": "LOCAL",
        "is_private": True
    }
