import re
from datetime import date
from django.core.validators import validate_email
from bson import ObjectId
from bson.errors import InvalidId
import phonenumbers
from django.core.exceptions import ValidationError
from urllib.parse import urlparse
from datetime import datetime, date





# =========================
# 🔹 REGEX & CONSTANTS
# =========================

PHONE_REGEX = re.compile(r'^\+?[1-9]\d{7,14}$')
NAME_REGEX = re.compile(r"^[A-Za-z .'-]{2,50}$")

GENDER_CHOICES = {"male", "female", "other"}
MIN_AGE = 13
MAX_AGE = 120


PASSWORD_REGEX = re.compile(
    r"""
    ^(?=.*[a-z])        # at least one lowercase
     (?=.*[A-Z])        # at least one uppercase
     (?=.*\d)           # at least one digit
     (?=.*[@$!%*?&^#()_\-+=\[\]{};:'",.<>\/\\|`~])  # special char
     .{8,}$             # minimum 8 characters
    """,
    re.VERBOSE
)

# =========================
# 🔹 BASIC FIELD VALIDATORS
# =========================

def validate_name(value: str, field="name"):
    if not value:
        raise ValidationError(f"{field} is required")
    if not NAME_REGEX.match(value):
        raise ValidationError(f"Invalid {field}")


def validate_email_field(value: str, field="email"):
    try:
        validate_email(value)
    except ValidationError:
        raise ValidationError(f"Invalid {field}")


def validate_phone(value: str, field="phone"):
    if not PHONE_REGEX.match(value):
        raise ValidationError(f"Invalid {field}")


def validate_gender(value: str):
    if value not in GENDER_CHOICES:
        raise ValidationError("Gender must be male, female or other")



def validate_dob(value):
    if isinstance(value, str):
        # Adjust the format '%Y-%m-%d' to match how your date is sent (e.g., '1990-01-01')
        value = datetime.strptime(value, "%Y-%m-%d").date()
        
    today = date.today()
    age = (today - value).days // 365

    if age < MIN_AGE:
        raise ValidationError("User must be at least 13 years old")
    if age > MAX_AGE:
        raise ValidationError("Invalid date of birth")


def validate_otp(value: str):
    if not value:
        raise ValidationError("OTP is required")
    if not value.isdigit():
        raise ValidationError("OTP must be numeric")
    if len(value) != 6:
        raise ValidationError("OTP must be exactly 6 digits")


def validate_user_id(value):
    if not value:
        raise ValidationError("User ID is Required")
    try:
        user_object_id = ObjectId(value)
    except InvalidId:
        raise ValidationError("Invalid User ID")



def validate_identifier(identifier):
    """
    Validates login identifier.
    Allowed:
    - Email
    - Phone number (international format)

    Returns:
    - 'email' or 'phone' (useful for login logic)

    Raises:
    - ValidationError if invalid
    """

    if not identifier:
        raise ValidationError("Identifier is required")

    identifier = identifier.strip()

    # Try email
    if "@" in identifier:
        validate_email_field(identifier)
        return "email"

    # Try phone
    
    if PHONE_REGEX.match(identifier):
        return "phone"

    raise ValidationError("Enter a valid email or phone number")


def validate_password(password: str):
    """
    Password rules:
    - Min 8 characters
    - At least 1 uppercase letter
    - At least 1 lowercase letter
    - At least 1 number
    - At least 1 special character
    """

    if not password:
        raise ValidationError("Password is required")

    if not PASSWORD_REGEX.match(password):
        raise ValidationError(
            "Password must be at least 8 characters long and include "
            "uppercase, lowercase, number, and special character"
        )




def validate_country_code(code):

    try:
        clean_code = int(str(code).strip())
        regions = phonenumbers.region_code_for_country_code(clean_code)
        
        if regions and 'ZZ' not in regions:
            return True, None
        
        raise ValidationError(
                " Enter a valid country or phone number"
            )
    except ValueError:
        raise ValidationError(
                " Enter a valid country or phone number"
            )






class OtpPurpose:
    LOGIN = "login"
    EMAIL_VERIFICATION = "email_verification"
    PHONE_VERIFICATION = "phone_verification"
    RECOVERY_EMAIL = "recovery_email"
    PASSWORD_RESET = "password_reset"
    ACCOUNT_RECOVERY = "account_recovery"


OTP_PURPOSES = {
    OtpPurpose.LOGIN,
    OtpPurpose.EMAIL_VERIFICATION,
    OtpPurpose.PHONE_VERIFICATION,
    OtpPurpose.RECOVERY_EMAIL,
    OtpPurpose.PASSWORD_RESET,
    OtpPurpose.ACCOUNT_RECOVERY,
}


def validate_otp_purpose(purpose: str):
    """
    Validates and normalizes OTP purpose.
    Input must be lowercase.
    Returns normalized purpose string.
    """

    if not purpose:
        raise ValidationError("OTP purpose is required")

    purpose = purpose.strip().lower()

    if purpose not in OTP_PURPOSES:
        raise ValidationError(
            f"Invalid OTP purpose. Allowed values: {', '.join(OTP_PURPOSES)}"
        )

    return purpose


# =========================
# 🔹 LOGIC / STATE VALIDATORS
# =========================

def validate_verification_state(data: dict):
    errors = {}

    if data.get("primaryPhoneVerified") and not data.get("phone_number"):
        errors["primaryPhoneVerified"] = "Phone number required for verification"

    if data.get("primary_email_verified") and not data.get("primary_email"):
        errors["primary_email_verified"] = "A primary email address must be provided before it can be verified"

    if data.get("recovery_phone_verified") and not data.get("recovery_phone"):
        errors["recovery_phone_verified"] = "A recovery phone number must be provided before it can be verified"

    if data.get("recovery_email_verified") and not data.get("recovery_email"):
        errors["recovery_email_verified"] = "A recovery email address must be provided before it can be verified"

    if data.get("email") and data.get("recovery_email"):
        if data["email"] == data["recovery_email"]:
            errors["recovery_email"] = "Recovery email must differ from primary"

    if data.get("phone_number") and data.get("recovery_phone"):
        if data["phone_number"] == data["recovery_phone"]:
            errors["recovery_phone"] = "Recovery phone must differ from primary"

    if data.get("isVerified"):
        if not (
            data.get("primaryPhoneVerified") or
            data.get("primary_email_verified")
        ):
            errors["isVerified"] = (
                "Account cannot be verified without verified email or phone"
            )

    if errors:
        raise ValidationError(errors)


# =========================
# 🔹 MASTER ACCOUNT VALIDATOR
# =========================

def validate_account_payload(data: dict):
    """
    🔥 One function to validate EVERYTHING
    Use this in:
    - DRF serializers
    - Signals
    - Services
    - Multi DB sync
    """

    # Required fields
    validate_name(data.get("first_name"), "first_name")
    validate_email_field(data.get("primary_email"))
    validate_phone(data.get("phone_number"))

    # Optional fields
    if data.get("last_name"):
        validate_name(data["last_name"], "last_name")

    if data.get("phone_number"):
        validate_country_code(data["country_code"])
        
    if data.get("primary_email"):
        validate_email_field(data["primary_email"], "primary_email")

    if data.get("recovery_email"):
        validate_email_field(data["recovery_email"], "recovery_email")
        

    if data.get("recovery_phone"):
        validate_phone(data["recovery_phone"], "recovery_phone")
        validate_country_code(data["recovery_phone_country_code"])        

    if data.get("gender"):
        validate_gender(data["gender"])

    if data.get("dob"):
        validate_dob(data["dob"])

    validate_verification_state(data)

    return True




DATE_FORMAT = "%Y-%m-%d"


def parse_date_safe(date_str: str, field_name: str) -> datetime:
    try:
        return datetime.strptime(date_str, DATE_FORMAT)
    except ValueError:
        raise ValidationError({
            field_name: f"Invalid date format. Expected YYYY-MM-DD."
        })


def validate_date_range(
    *,
    from_date: str | None,
    to_date: str | None,
    allow_future: bool = False
):
    """
    Validates date range and returns parsed datetimes (UTC midnight).

    Rules:
    - both from & to must be present together
    - from <= to
    - to <= today (unless allow_future=True)
    """

    if from_date and not to_date:
        raise ValidationError({"to": "This field is required when 'from' is provided."})

    if to_date and not from_date:
        raise ValidationError({"from": "This field is required when 'to' is provided."})

    if not from_date and not to_date:
        return None, None

    start = parse_date_safe(from_date, "from")
    end = parse_date_safe(to_date, "to")

    start = start.replace(hour=0, minute=0, second=0, microsecond=0)
    end = end.replace(hour=0, minute=0, second=0, microsecond=0)

    if start > end:
        raise ValidationError({
            "date_range": "'from' date cannot be greater than 'to' date."
        })

    today = datetime.utcnow().replace(hour=0, minute=0, second=0, microsecond=0)

    if not allow_future and end > today:
        raise ValidationError({
            "to": "Date cannot be in the future."
        })

    return start, end




ALLOWED_TICKET_CATEGORIES = {
    "ACCOUNT",
    "PAYMENT",
    "SECURITY",
    "TECHNICAL",
    "FEATURE_REQUEST",
    "BUG_REPORT",
    "OTHER"
}


def validate_ticket_payload(data):
    required_fields = ["name", "email", "phone", "category", "subject", "description"]

    for field in required_fields:
        if not data.get(field):
            raise ValidationError({"error": f"{field} is required"})

    if data["category"].upper() not in ALLOWED_TICKET_CATEGORIES:
        raise ValidationError({
            "error": f"Invalid category. Allowed: {', '.join(ALLOWED_TICKET_CATEGORIES)}"
        })

    if len(data["subject"].strip()) < 5:
        raise ValidationError({"error": "Subject too short"})

    if len(data["description"].strip()) < 10:
        raise ValidationError({"error": "Description too short"})


    if  data["email"]:
        validate_email(data["email"])

    if  data["phone"]:
        validate_phone(data["phone"])





SEARCH_QUERY_REGEX = re.compile(r"^[A-Za-z0-9 @]+$")

def validate_search_query(value: str):
    if not value:
        raise ValidationError("Search query is required")

    if not isinstance(value, str):
        raise ValidationError("Invalid search query")

    value = value.strip()

    if not value:
        raise ValidationError("Search query cannot be empty")

    if not SEARCH_QUERY_REGEX.match(value):
        raise ValidationError(
            "Search query contains invalid characters. "
            "Only letters, numbers, spaces, and @ are allowed."
        )






# Allows letters, numbers, spaces, and common punctuation
DESCRIPTION_REGEX = re.compile(
    r"^[A-Za-z0-9\s.,:;!?()'\"/@#\-–—_]+$"
)

def validate_description(value: str):
    if not value:
        raise ValidationError("Description is required")

    if not isinstance(value, str):
        raise ValidationError("Invalid description")

    value = value.strip()

    if not value:
        raise ValidationError("Description cannot be empty")

    if len(value) < 10:
        raise ValidationError("Description must be at least 10 characters long")

    if len(value) > 1000:
        raise ValidationError("Description must not exceed 1000 characters")

    if not DESCRIPTION_REGEX.match(value):
        raise ValidationError(
            "Description contains invalid characters or emojis"
        )




# No emojis, only valid URL characters
URL_REGEX = re.compile(
    r"^[A-Za-z0-9:/?#\[\]@!$&'()*+,;=.\-_~%]+$"
)

def validate_url(value: str):
    if not value:
        raise ValidationError("URL is required")

    if not isinstance(value, str):
        raise ValidationError("Invalid URL")

    value = value.strip()

    if not value:
        raise ValidationError("URL cannot be empty")

    # Reject emojis / invalid characters
    if not URL_REGEX.match(value):
        raise ValidationError("URL contains invalid characters")

    parsed = urlparse(value)

    # Scheme validation
    if parsed.scheme not in ("http", "https"):
        raise ValidationError("Only http and https URLs are allowed")

    # Domain validation
    if not parsed.netloc:
        raise ValidationError("Invalid URL format")

    # Prevent javascript/data URLs
    if parsed.scheme in ("javascript", "data"):
        raise ValidationError("Unsafe URL")


# ------------- Usage example ---------------------------



# from rest_framework.views import APIView
# from django.core.exceptions import ValidationError
# from core.validators.account_validators import validate_otp
# from core.utils.api_response import success, error

# class VerifyOTPView(APIView):

#     def post(self, request):
#         otp = request.data.get("otp")

#         try:
#             validate_otp(otp)
#         except ValidationError as e:
#             return error(
#                 message="Invalid OTP",
#                 errors=e.messages
#             )

#         # 🔐 verify OTP logic here

#         return success(message="OTP verified successfully")
